package com.aps.controller.gantt;

import cn.hutool.core.bean.BeanUtil;
import com.aps.common.util.ParamValidator;
import com.aps.common.util.R;
import com.aps.entity.Algorithm.Chromosome;
import com.aps.entity.basic.MaintenanceWindow;
import com.aps.service.plan.PlanResultService;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.responses.ApiResponses;
import io.swagger.v3.oas.annotations.tags.Tag;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import java.util.List;
import java.util.Map;

@Slf4j
@RestController
@RequestMapping("/maintenance")
@Tag(name = "维护窗口管理", description = "维护窗口相关API")
public class MaintenanceWindowController {

    @Autowired
    private PlanResultService planResultService;

    @PostMapping("/add")
    @Operation(summary = "添加维护窗口", description = "为指定机器添加维护窗口",
            requestBody = @io.swagger.v3.oas.annotations.parameters.RequestBody(
                    description = "维护窗口参数",
                    content = @io.swagger.v3.oas.annotations.media.Content(
                            mediaType = "application/json",
                            examples = @io.swagger.v3.oas.annotations.media.ExampleObject(
                                    name = "添加维护窗口示例",
                                    value = "{\n  \"sceneId\": \"B571EF6682DB463AB2977B1055A74112\",\n  \"machineId\": 3402,\n  \"maintenanceWindow\": {\n    \"startTime\": \"2025-11-03T07:36:00.000Z\",\n    \"endTime\": \"2025-11-03T09:36:00.000Z\",\n    \"reason\": \"定期维护\"\n  }\n}"
                            )
                    )
            )
    )
    public R<String> addMaintenanceWindow(@RequestBody Map<String, Object> params) {
        String sceneId = ParamValidator.getString(params, "sceneId", "场景ID");
        Long machineId = ParamValidator.getLong(params, "machineId", "机器ID");
        MaintenanceWindow maintenanceWindow = BeanUtil.toBean(params.get("maintenanceWindow"), MaintenanceWindow.class);
        Chromosome result = planResultService.AddMaintenanceWindow(sceneId, machineId, maintenanceWindow);
        return R.ok("添加成功");
    }

    @PostMapping("/delete")
    @Operation(summary = "删除维护窗口", description = "删除指定机器的维护窗口",
            requestBody = @io.swagger.v3.oas.annotations.parameters.RequestBody(
                    description = "删除维护窗口请求参数",
                    content = @io.swagger.v3.oas.annotations.media.Content(
                            mediaType = "application/json",
                            examples = @io.swagger.v3.oas.annotations.media.ExampleObject(
                                    name = "删除维护窗口示例",
                                    value = "{\n  \"sceneId\": \"scene001\",\n  \"machineId\": 1,\n  \"maintenanceId\": \"maint001\"\n}"
                            )
                    )
            )
    )
    public R<String> delMaintenanceWindow(@RequestBody Map<String, Object> params) {
        String sceneId = ParamValidator.getString(params, "sceneId", "场景ID");
        Long machineId = ParamValidator.getLong(params, "machineId", "机器ID");
        String maintenanceId = ParamValidator.getString(params, "maintenanceId", "维护窗口ID");

        Chromosome result = planResultService.DelMaintenanceWindow(sceneId, machineId, maintenanceId);
        return R.ok("删除成功");
    }

    @PostMapping("/batchDelete")
    @Operation(summary = "批量删除维护窗口", description = "批量删除指定机器的维护窗口",
            requestBody = @io.swagger.v3.oas.annotations.parameters.RequestBody(
                    description = "批量删除维护窗口请求参数",
                    content = @io.swagger.v3.oas.annotations.media.Content(
                            mediaType = "application/json",
                            examples = @io.swagger.v3.oas.annotations.media.ExampleObject(
                                    name = "批量删除维护窗口示例",
                                    value = "{\n  \"sceneId\": \"scene001\",\n  \"machineId\": 1,\n  \"maintenanceIds\": [\"maint001\", \"maint002\", \"maint003\"]\n}"
                            )
                    )
            )
    )
    public R<String> delMaintenanceWindowBatch(@RequestBody Map<String, Object> params) {
        String sceneId = ParamValidator.getString(params, "sceneId", "场景ID");
        Long machineId = ParamValidator.getLong(params, "machineId", "机器ID");
        List<String> maintenanceIds = (List<String>) params.get("maintenanceIds");


        Chromosome result = planResultService.DelMaintenanceWindowBatch(sceneId, machineId, maintenanceIds);
        return R.ok("批量删除成功");
    }

    @PostMapping("/get")
    @Operation(summary = "获取维护窗口", description = "获取指定机器的维护窗口列表",
            requestBody = @io.swagger.v3.oas.annotations.parameters.RequestBody(
                    description = "获取维护窗口请求参数",
                    content = @io.swagger.v3.oas.annotations.media.Content(
                            mediaType = "application/json",
                            examples = @io.swagger.v3.oas.annotations.media.ExampleObject(
                                    name = "获取维护窗口示例",
                                    value = "{\n  \"sceneId\": \"scene001\",\n  \"machineId\": 1\n}"
                            )
                    )
            )
    )
    @ApiResponses({
            @ApiResponse(responseCode = "200", description = "成功获取维护窗口列表")
    })
    public R<List<MaintenanceWindow>> getMaintenanceWindow(@RequestBody Map<String, Object> params) {
        String sceneId = ParamValidator.getString(params, "sceneId", "场景ID");
        Long machineId = ParamValidator.getLong(params, "machineId", "机器ID");

        List<MaintenanceWindow> result = planResultService.GetMaintenanceWindow(sceneId, machineId, null);
        return R.ok(result);
    }
}