package com.aps.service.Algorithm;

import com.aps.entity.Algorithm.Chromosome;
import com.aps.entity.Algorithm.GlobalOperationInfo;
import com.aps.entity.Algorithm.ScheduleParams;
import com.aps.entity.basic.Entry;
import com.aps.entity.basic.GlobalParam;
import com.aps.entity.basic.MachineOption;

import java.util.*;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

/**
 * 作者：佟礼
 * 时间：2025-11-21
 */
public class Initialization {
    private static List<Entry> allOperations;
    private static GlobalParam _globalParam;

    public Initialization(GlobalParam globalParam,List<Entry> allOperations) {
        Initialization.allOperations = allOperations;
        _globalParam= globalParam;
    }
    /**
     * 预生成全局工序列表（按“订单0→订单1→…+订单内工序1→2→…”排序，分配GlobalOpId）
     */
    public List<GlobalOperationInfo> generateGlobalOpList() {
        List<GlobalOperationInfo> globalOpList = new ArrayList<>();
        int globalOpId = 0;
        for (Entry op : allOperations) {
            GlobalOperationInfo info = new GlobalOperationInfo();
            info.setGlobalOpId(globalOpId);
            info.setGroupId(op.getGroupId());
            info.setSequence(op.getSequence());
            info.setOp(op);
            globalOpList.add(info);
            globalOpId++;
        }
        return globalOpList;
    }
    /**
     * 生成初始种群
     */
    public List<Chromosome> generateInitialPopulation(ScheduleParams param, List<GlobalOperationInfo> globalOpList) {
        List<Chromosome> population = new ArrayList<>();
        // 按比例生成不同类型个体（GS:40%, LS:40%, RS:20%）
        int gsCount = (int) (param.getPopulationSize() * param.getGsRatio());
        int lsCount =gsCount+ (int) (param.getPopulationSize() * param.getLsRatio());
        int rsCount = param.getPopulationSize() - gsCount - lsCount;
int populationSize=param.getPopulationSize();
        // 并行循环：对应 Parallel.For(0, PopulationSize, i => { ... })
        IntStream.range(0, populationSize)
                .parallel() // 开启并行
                .forEach(i -> {
                    Chromosome chromo = new Chromosome(); // 初始化染色体


                    // 全局选择（GS）：按GlobalOpId顺序生成MachineSelection
                    if (i < gsCount) {
                        generateGSChromosome(chromo,globalOpList); // 对应 C# GenerateGSChromosome
                    } else if (i < lsCount) {
                        // 局部选择（LS）：按GlobalOpId顺序生成MachineSelection（仅负载计算范围不同）
                        generateLSChromosome(chromo,globalOpList);
                    } else {
                        //  随机选择（RS）：按GlobalOpId顺序生成MachineSelection（仅机器选择随机）
                        generateRSChromosome(chromo,globalOpList);
                    }

                    population.add(chromo); // 赋值到数组，线程安全（数组索引唯一）
                });





        return population;
    }
    /**
     * 全局选择（GS）生成染色体（按GlobalOpId顺序生成MachineSelection）
     */
    private Chromosome generateGSChromosome(Chromosome chromosome,List<GlobalOperationInfo> globalOpList) {


        Map<Long, Double> machineLoad = new HashMap<>();
       // int[] machineLoad = new int[machineCount + 1]; // 设备负载（1-based，设备ID从1开始）
        List<Integer> ms = new ArrayList<>(); // MachineSelection（顺序=GlobalOpId顺序）
        List<Integer> os = new ArrayList<>(); // OperationSequencing
        List<Integer> osp = new ArrayList<>(); // 相同优先级OperationSequencing
        double prevPriority =(double) -1;
        Random rnd = new Random();

        // 步骤1：按GlobalOpId顺序为每道工序选择机器
        for (GlobalOperationInfo globalOp : globalOpList) {
            Entry op = globalOp.getOp();
            int groupId = globalOp.getGroupId();

            if (prevPriority != op.getPriority()) {
                prevPriority = op.getPriority();
                if (!osp.isEmpty()) {
                    // 打乱工序排序部分
                    Collections.shuffle(osp);
                    os.addAll(osp);
                    osp.clear();
                }
            }

            // 选择“设备当前负载+工序加工时间”最小的机器（全局负载均衡）
            List<MachineOption> optionalMachines = op.getMachineOptions();

            MachineOption minLoadMachine = optionalMachines.stream()
                    .min(Comparator.comparingDouble(m ->machineLoad.getOrDefault(m.getMachineId(), (double)0) + m.getProcessingTime()))
                    .orElseThrow(() -> new NoSuchElementException("MachineOption not found for machine: " ));


            OptionalInt index = IntStream.range(0, optionalMachines.size())
                    .filter(i -> minLoadMachine.getMachineId()==optionalMachines.get(i).getMachineId())
                    .findFirst();

            // 计算该机器在“可选机器列表”中的顺序号（1-based）
            int machineSeq =index.orElse(0)+1 ;//findIndex(optionalMachines, minLoadMachine.getMachineId()) + 1;
            ms.add(machineSeq);

            // 更新设备负载
            machineLoad.put(minLoadMachine.getMachineId(),
                    machineLoad.getOrDefault(minLoadMachine.getMachineId(), (double)0) + minLoadMachine.getProcessingTime());

            osp.add(groupId);
        }

        if (!osp.isEmpty()) {
            Collections.shuffle(osp, rnd);
            os.addAll(osp);
            osp.clear();
        }

        // 步骤2：打乱工序排序部分（OS）
        chromosome.setOperationSequencing(shuffleWithPriority(os));
        chromosome.setMachineSelection(ms);

        return chromosome;
    }

    /**
     * 局部选择（LS）生成染色体（按GlobalOpId顺序，每个订单重新初始化负载）
     */
    private Chromosome generateLSChromosome(Chromosome chromosome,List<GlobalOperationInfo> globalOpList) {


        List<Integer> ms = new ArrayList<>();
        List<Integer> os = new ArrayList<>();
        Random rnd = new Random();
        int currentOrderId = -1;
        Map<Long, Double> machineLoad = new HashMap<>();
        List<Integer> osp = new ArrayList<>();
        double prevPriority = -1;

        for (GlobalOperationInfo globalOp : globalOpList) {
            Entry op = globalOp.getOp();
            int groupId = globalOp.getGroupId();

            if (prevPriority != op.getPriority()) {
                prevPriority = op.getPriority();
                if (!osp.isEmpty()) {
                    Collections.shuffle(osp, rnd);
                    os.addAll(osp);
                    osp.clear();
                }
            }

            // 若切换到新订单，重置设备负载
            if (groupId != currentOrderId) {
                machineLoad.clear();
                currentOrderId = groupId;
            }

            // 选择“当前订单内设备负载+加工时间”最小的机器
            List<MachineOption> optionalMachines = op.getMachineOptions();


            MachineOption minLoadMachine = optionalMachines.stream()
                    .min(Comparator.comparingDouble(m -> machineLoad.getOrDefault(m.getMachineId(), (double)0) + m.getProcessingTime()))
                    .orElseThrow(() -> new NoSuchElementException("MachineOption not found for machine: " ));


            OptionalInt index = IntStream.range(0, optionalMachines.size())
                    .filter(i -> minLoadMachine.getMachineId()==optionalMachines.get(i).getMachineId())
                    .findFirst();


            int machineSeq = index.orElse(0) + 1;
            ms.add(machineSeq);

          double maLoad=  machineLoad.getOrDefault(minLoadMachine.getMachineId(),(double)0);
            // 更新设备负载
            machineLoad.put(minLoadMachine.getMachineId(),
                    maLoad+ minLoadMachine.getProcessingTime());

            osp.add(groupId);
        }

        if (!osp.isEmpty()) {
            Collections.shuffle(osp, rnd);
            os.addAll(osp);
            osp.clear();
        }

        chromosome.setOperationSequencing(shuffleWithPriority(os));
        chromosome.setMachineSelection(ms);

        return chromosome;
    }
    /**
     * 随机选择（RS）生成染色体（按GlobalOpId顺序，随机选择机器）
     */
    private Chromosome generateRSChromosome(Chromosome chromosome,List<GlobalOperationInfo> globalOpList) {


        List<Integer> ms = new ArrayList<>();
        List<Integer> os = new ArrayList<>();
        Random rnd = new Random();
        List<Integer> osp = new ArrayList<>();
        double prevPriority = -1;

        for (GlobalOperationInfo globalOp : globalOpList) {
            Entry op = globalOp.getOp();
            int groupId = globalOp.getGroupId();

            if (prevPriority != op.getPriority()) {
                prevPriority = op.getPriority();
                if (!osp.isEmpty()) {
                    Collections.shuffle(osp, rnd);
                    os.addAll(osp);
                    osp.clear();
                }
            }

            // 随机选择一道工序的可选机器（1-based顺序号）
            List<MachineOption> optionalMachines = op.getMachineOptions();
            int randomSeq = rnd.nextInt(optionalMachines.size()) + 1;
            ms.add(randomSeq);

            osp.add(groupId);
        }

        if (!osp.isEmpty()) {
            Collections.shuffle(osp, rnd);
            os.addAll(osp);
            osp.clear();
        }

        chromosome.setOperationSequencing(shuffleWithPriority(os));
        chromosome.setMachineSelection(ms);

        return chromosome;
    }

    /**
     * 按优先级加权打乱工序排序（高优先级订单的工序更可能靠前）
     */
    private List<Integer> shuffleWithPriority(List<Integer> os) {

        if (!_globalParam.isIsBreakPriority()) {
            return new ArrayList<>(os);
        }

        Random rnd = new Random();
        double maxPriority = allOperations.stream()
                .mapToDouble(Entry::getPriority)
                .max()
                .orElse(0) + 1;

        // 为每个工序分配权重：低优先级订单的工序权重更高
        List<WeightedGroup> weightedOs = os.stream()
                .map(groupId -> {
                    Entry entry = allOperations.stream()
                            .filter(o -> o.getGroupId() == groupId)
                            .findFirst().orElse(null);

                    return new WeightedGroup(groupId, maxPriority - entry.getPriority());
                })
                .collect(Collectors.toList());

        // 按权重随机排序（权重越高，被选中的概率越大）
        List<Integer> shuffled = new ArrayList<>();
        while (!weightedOs.isEmpty()) {
            double totalWeight = weightedOs.stream().mapToDouble(WeightedGroup::getWeight).sum();
            int randomWeight = rnd.nextInt((int)totalWeight);
            int cumulativeWeight = 0;
            int selectedIndex = -1;

            for (int i = 0; i < weightedOs.size(); i++) {
                cumulativeWeight += weightedOs.get(i).getWeight();
                if (cumulativeWeight >= randomWeight) {
                    selectedIndex = i;
                    break;
                }
            }

            shuffled.add(weightedOs.get(selectedIndex).getGroupId());
            weightedOs.remove(selectedIndex);
        }

        return shuffled;
    }
    /**
     * 辅助类：用于权重排序
     */
    private static class WeightedGroup {
        private final int groupId;
        private final double weight;

        public WeightedGroup(int groupId, double weight) {
            this.groupId = groupId;
            this.weight = weight;
        }

        public int getGroupId() {
            return groupId;
        }

        public double getWeight() {
            return weight;
        }
    }

}
