package com.aps.service.common;

import com.aps.entity.common.ConditionEntity;
import com.aps.entity.common.ConditionEnum;
import com.aps.entity.common.EntityConfig;
import com.aps.entity.common.Paged;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.jdbc.core.BeanPropertyRowMapper;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.jdbc.core.namedparam.MapSqlParameterSource;
import org.springframework.jdbc.core.namedparam.NamedParameterJdbcTemplate;
import org.springframework.stereotype.Service;

import java.util.*;

@Service
public class DatabaseQueryService {

    @Autowired
    private JdbcTemplate jdbcTemplate;

    @Autowired
    private NamedParameterJdbcTemplate namedParameterJdbcTemplate;

    /**
     * 查询数据库数据（分页+条件）- 修复Oracle分页问题。
     * 当 paged.groupName 有值时按分组字段 GROUP BY，返回分组列 + count（每组条数）。
     */
    public Map<String, Object> queryDatabaseDataWithConditions(EntityConfig config, Paged paged) {
        String tableName = config.getTableName();
        String groupName = paged.getGroupName();
        boolean isGroupBy = groupName != null && !groupName.trim().isEmpty();

        if (isGroupBy) {
            return queryDatabaseDataWithGroupBy(config, paged, tableName);
        }

        // 构建基础SQL和参数
        StringBuilder whereClause = new StringBuilder(" WHERE 1=1");
        MapSqlParameterSource params = new MapSqlParameterSource();

        // 处理条件
        int paramIndex = 1;
        if (paged.getConditions() != null && !paged.getConditions().isEmpty()) {
            boolean isAnd = paged.getAnd() != null && paged.getAnd();
            String logicalOperator = isAnd ? " AND " : " OR ";

            // 构建条件列表
            List<String> conditionSqlList = new ArrayList<>();
            for (ConditionEntity condition : paged.getConditions()) {
                String conditionSql = buildConditionSql(condition, params, paramIndex);
                if (conditionSql != null) {
                    conditionSqlList.add(conditionSql);
                    paramIndex++;
                }
            }

            // 如果有条件，使用指定的逻辑运算符连接
            if (!conditionSqlList.isEmpty()) {
                whereClause.append(" AND (").append(String.join(logicalOperator, conditionSqlList)).append(")");
            }
        }

        // 构建排序
        String orderBy = buildOrderBy(paged);

        // 分页参数
        int page = paged.getPageIndex() != null ? paged.getPageIndex() : 1;
        int size = paged.getPageSize() != null ? paged.getPageSize() : 10;

        // 返回列：fields 有值时只查指定列，否则 SELECT *
        String selectClause = "*";
        if (paged.getFields() != null && !paged.getFields().trim().isEmpty()) {
            List<String> cols = parseFieldListToDbColumns(paged.getFields());
            if (!cols.isEmpty()) {
                selectClause = String.join(", ", cols);
            }
        }
        boolean useFieldsOnly = !"*".equals(selectClause);

        // 先查询总数
        String countSql = "SELECT COUNT(*) FROM " + tableName + whereClause.toString();
        Integer total;
        try {
            if (params.getValues().isEmpty()) {
                total = jdbcTemplate.queryForObject(countSql, Integer.class);
            } else {
                total = namedParameterJdbcTemplate.queryForObject(countSql, params, Integer.class);
            }
        } catch (Exception e) {
            total = 0;
        }

        // 查询数据（使用Oracle分页语法）
        List<?> records = Collections.emptyList();
        if (total != null && total > 0) {
            String dataSql = buildOraclePaginationSql(tableName, whereClause.toString(), orderBy, page, size, selectClause);
            try {
                if (useFieldsOnly) {
                    // 指定了 fields 时只返回这些列，用 Map 列表（key 转小写）
                    List<Map<String, Object>> rows = params.getValues().isEmpty()
                            ? jdbcTemplate.queryForList(dataSql)
                            : namedParameterJdbcTemplate.queryForList(dataSql, params);
                    records = normalizeMapKeysToLower(rows);
                } else if (config.getEntityClass() != null) {
                    if (params.getValues().isEmpty()) {
                        records = jdbcTemplate.query(dataSql, new BeanPropertyRowMapper<>(config.getEntityClass()));
                    } else {
                        records = namedParameterJdbcTemplate.query(dataSql, params, new BeanPropertyRowMapper<>(config.getEntityClass()));
                    }
                } else {
                    if (params.getValues().isEmpty()) {
                        records = jdbcTemplate.queryForList(dataSql);
                    } else {
                        records = namedParameterJdbcTemplate.queryForList(dataSql, params);
                    }
                }
            } catch (Exception e) {
                throw new RuntimeException("查询数据失败: " + e.getMessage(), e);
            }
        }

        Map<String, Object> result = new HashMap<>();
        result.put("records", records);
        result.put("totalCount", total != null ? total : 0);
        result.put("pageIndex", page);
        result.put("size", size);

        return result;
    }

    /**
     * 按 groupName 分组查询：返回分组列 + count（每组条数），支持分页。
     */
    private Map<String, Object> queryDatabaseDataWithGroupBy(EntityConfig config, Paged paged, String tableName) {
        StringBuilder whereClause = new StringBuilder(" WHERE 1=1");
        MapSqlParameterSource params = new MapSqlParameterSource();
        int paramIndex = 1;
        if (paged.getConditions() != null && !paged.getConditions().isEmpty()) {
            boolean isAnd = paged.getAnd() != null && paged.getAnd();
            String logicalOperator = isAnd ? " AND " : " OR ";
            List<String> conditionSqlList = new ArrayList<>();
            for (ConditionEntity condition : paged.getConditions()) {
                String conditionSql = buildConditionSql(condition, params, paramIndex);
                if (conditionSql != null) {
                    conditionSqlList.add(conditionSql);
                    paramIndex++;
                }
            }
            if (!conditionSqlList.isEmpty()) {
                whereClause.append(" AND (").append(String.join(logicalOperator, conditionSqlList)).append(")");
            }
        }

        List<String> groupCols = parseFieldListToDbColumns(paged.getGroupName());
        if (groupCols.isEmpty()) {
            Map<String, Object> empty = new HashMap<>();
            empty.put("records", Collections.emptyList());
            empty.put("totalCount", 0);
            empty.put("pageIndex", paged.getPageIndex() != null ? paged.getPageIndex() : 1);
            empty.put("size", paged.getPageSize() != null ? paged.getPageSize() : 10);
            return empty;
        }

        String groupColsStr = String.join(", ", groupCols);
        String orderBy = buildOrderBy(paged);
        int page = paged.getPageIndex() != null ? paged.getPageIndex() : 1;
        int size = paged.getPageSize() != null ? paged.getPageSize() : 10;

        String countSql = "SELECT COUNT(*) FROM (SELECT " + groupColsStr + " FROM " + tableName + whereClause + " GROUP BY " + groupColsStr + ")";
        Integer total;
        try {
            if (params.getValues().isEmpty()) {
                total = jdbcTemplate.queryForObject(countSql, Integer.class);
            } else {
                total = namedParameterJdbcTemplate.queryForObject(countSql, params, Integer.class);
            }
        } catch (Exception e) {
            total = 0;
        }

        List<Map<String, Object>> records = new ArrayList<>();
        if (total != null && total > 0) {
            String innerSql = "SELECT " + groupColsStr + ", COUNT(*) AS cnt FROM " + tableName + whereClause + " GROUP BY " + groupColsStr + orderBy;
            String dataSql = buildOraclePaginationSqlForGroupBy(innerSql, page, size);
            try {
                List<Map<String, Object>> rows = params.getValues().isEmpty()
                        ? jdbcTemplate.queryForList(dataSql)
                        : namedParameterJdbcTemplate.queryForList(dataSql, params);
                for (Map<String, Object> row : rows) {
                    Map<String, Object> out = new LinkedHashMap<>();
                    for (Map.Entry<String, Object> e : row.entrySet()) {
                        String key = e.getKey();
                        if ("CNT".equalsIgnoreCase(key) || "cnt".equals(key)) {
                            out.put("count", e.getValue());
                        } else {
                            out.put(key != null ? key.toLowerCase() : key, e.getValue());
                        }
                    }
                    records.add(out);
                }
            } catch (Exception e) {
                throw new RuntimeException("分组查询失败: " + e.getMessage(), e);
            }
        }

        Map<String, Object> result = new HashMap<>();
        result.put("records", records);
        result.put("totalCount", total != null ? total : 0);
        result.put("pageIndex", page);
        result.put("size", size);
        return result;
    }

    private String buildOraclePaginationSqlForGroupBy(String innerSql, int page, int size) {
        int startRow = (page - 1) * size + 1;
        int endRow = page * size;
        return "SELECT * FROM (SELECT a.*, ROWNUM rn FROM (" + innerSql + ") a WHERE ROWNUM <= " + endRow + ") WHERE rn >= " + startRow;
    }

    /**
     * 查询数据库数据列表（不分页）。当 paged.groupName 有值时按分组字段 GROUP BY，返回分组列 + count。
     */
    public List<Object> queryDatabaseDataList(EntityConfig config, Paged paged) {
        String tableName = config.getTableName();
        String groupName = paged.getGroupName();
        boolean isGroupBy = groupName != null && !groupName.trim().isEmpty();

        if (isGroupBy) {
            return queryDatabaseDataListWithGroupBy(config, paged, tableName);
        }

        StringBuilder whereClause = new StringBuilder(" WHERE 1=1");
        MapSqlParameterSource params = new MapSqlParameterSource();

        // 处理条件
        int paramIndex = 1;
        if (paged.getConditions() != null && !paged.getConditions().isEmpty()) {
            boolean isAnd = paged.getAnd() != null && paged.getAnd();
            String logicalOperator = isAnd ? " AND " : " OR ";

            // 构建条件列表
            List<String> conditionSqlList = new ArrayList<>();
            for (ConditionEntity condition : paged.getConditions()) {
                String conditionSql = buildConditionSql(condition, params, paramIndex);
                if (conditionSql != null) {
                    conditionSqlList.add(conditionSql);
                    paramIndex++;
                }
            }

            // 如果有条件，使用指定的逻辑运算符连接
            if (!conditionSqlList.isEmpty()) {
                whereClause.append(" AND (").append(String.join(logicalOperator, conditionSqlList)).append(")");
            }
        }

        // 构建排序
        String orderBy = buildOrderBy(paged);

        // 返回列：fields 有值时只查指定列
        String selectClause = "*";
        if (paged.getFields() != null && !paged.getFields().trim().isEmpty()) {
            List<String> cols = parseFieldListToDbColumns(paged.getFields());
            if (!cols.isEmpty()) {
                selectClause = String.join(", ", cols);
            }
        }
        boolean useFieldsOnly = !"*".equals(selectClause);

        String sql = "SELECT " + selectClause + " FROM " + tableName + whereClause.toString() + orderBy;

        try {
            if (useFieldsOnly) {
                List<Map<String, Object>> rows = params.getValues().isEmpty()
                        ? jdbcTemplate.queryForList(sql)
                        : namedParameterJdbcTemplate.queryForList(sql, params);
                return new ArrayList<>(normalizeMapKeysToLower(rows));
            }
            if (config.getEntityClass() != null) {
                if (params.getValues().isEmpty()) {
                    return new ArrayList<>(jdbcTemplate.query(sql, new BeanPropertyRowMapper<>(config.getEntityClass())));
                } else {
                    return new ArrayList<>(namedParameterJdbcTemplate.query(sql, params, new BeanPropertyRowMapper<>(config.getEntityClass())));
                }
            } else {
                if (params.getValues().isEmpty()) {
                    return new ArrayList<>(jdbcTemplate.queryForList(sql));
                } else {
                    return new ArrayList<>(namedParameterJdbcTemplate.queryForList(sql, params));
                }
            }
        } catch (Exception e) {
            throw new RuntimeException("查询数据列表失败: " + e.getMessage(), e);
        }
    }

    /** 解析 "id,name" 为数据库列名列表（大写），供 SELECT / GROUP BY 使用 */
    private List<String> parseFieldListToDbColumns(String fieldList) {
        List<String> cols = new ArrayList<>();
        if (fieldList == null || fieldList.trim().isEmpty()) return cols;
        for (String part : fieldList.split(",")) {
            String trimmed = part.trim();
            if (!trimmed.isEmpty()) {
                cols.add(camelCaseToUnderScoreUpperCase(trimmed).toUpperCase());
            }
        }
        return cols;
    }

    private List<Map<String, Object>> normalizeMapKeysToLower(List<Map<String, Object>> rows) {
        List<Map<String, Object>> out = new ArrayList<>();
        for (Map<String, Object> row : rows) {
            Map<String, Object> m = new LinkedHashMap<>();
            for (Map.Entry<String, Object> e : row.entrySet()) {
                m.put(e.getKey() != null ? e.getKey().toLowerCase() : "", e.getValue());
            }
            out.add(m);
        }
        return out;
    }

    private List<Object> queryDatabaseDataListWithGroupBy(EntityConfig config, Paged paged, String tableName) {
        StringBuilder whereClause = new StringBuilder(" WHERE 1=1");
        MapSqlParameterSource params = new MapSqlParameterSource();
        int paramIndex = 1;
        if (paged.getConditions() != null && !paged.getConditions().isEmpty()) {
            boolean isAnd = paged.getAnd() != null && paged.getAnd();
            String logicalOperator = isAnd ? " AND " : " OR ";
            List<String> conditionSqlList = new ArrayList<>();
            for (ConditionEntity condition : paged.getConditions()) {
                String conditionSql = buildConditionSql(condition, params, paramIndex);
                if (conditionSql != null) {
                    conditionSqlList.add(conditionSql);
                    paramIndex++;
                }
            }
            if (!conditionSqlList.isEmpty()) {
                whereClause.append(" AND (").append(String.join(logicalOperator, conditionSqlList)).append(")");
            }
        }

        List<String> groupCols = parseFieldListToDbColumns(paged.getGroupName());
        if (groupCols.isEmpty()) {
            return new ArrayList<>();
        }
        String groupColsStr = String.join(", ", groupCols);
        String orderBy = buildOrderBy(paged);
        String sql = "SELECT " + groupColsStr + ", COUNT(*) AS cnt FROM " + tableName + whereClause + " GROUP BY " + groupColsStr + orderBy;

        try {
            List<Map<String, Object>> rows = params.getValues().isEmpty()
                    ? jdbcTemplate.queryForList(sql)
                    : namedParameterJdbcTemplate.queryForList(sql, params);
            List<Object> result = new ArrayList<>();
            for (Map<String, Object> row : rows) {
                Map<String, Object> out = new LinkedHashMap<>();
                for (Map.Entry<String, Object> e : row.entrySet()) {
                    String key = e.getKey();
                    if ("CNT".equalsIgnoreCase(key) || "cnt".equals(key)) {
                        out.put("count", e.getValue());
                    } else {
                        out.put(key != null ? key.toLowerCase() : key, e.getValue());
                    }
                }
                result.add(out);
            }
            return result;
        } catch (Exception e) {
            throw new RuntimeException("分组查询列表失败: " + e.getMessage(), e);
        }
    }

    /**
     * 根据ID查询单条数据
     */
    public Object queryDatabaseDataById(EntityConfig config, String id) {
        String tableName = config.getTableName();

        // 尝试不同的ID字段名（兼容大小写）
        String[] idFields = {"ID", "Id", "id"};

        for (String idField : idFields) {
            try {
                String sql = "SELECT * FROM " + tableName + " WHERE " + idField + " = ?";
                if (config.getEntityClass() != null) {
                    return jdbcTemplate.queryForObject(sql, new BeanPropertyRowMapper<>(config.getEntityClass()), id);
                } else {
                    return jdbcTemplate.queryForMap(sql, id);
                }
            } catch (Exception e) {
                // 继续尝试下一个字段名
            }
        }

        return null; // 找不到数据时返回null而不是抛出异常
    }

    /**
     * 构建Oracle分页SQL；selectClause 为 null 或空时使用 *。
     */
    private String buildOraclePaginationSql(String tableName, String whereClause, String orderBy, int page, int size, String selectClause) {
        int startRow = (page - 1) * size + 1;
        int endRow = page * size;
        String cols = (selectClause != null && !selectClause.trim().isEmpty()) ? selectClause : "*";

        StringBuilder sql = new StringBuilder();
        sql.append("SELECT * FROM (");
        sql.append("    SELECT a.*, ROWNUM rn FROM (");
        sql.append("        SELECT ").append(cols).append(" FROM ").append(tableName).append(whereClause).append(orderBy);
        sql.append("    ) a WHERE ROWNUM <= ").append(endRow);
        sql.append(") WHERE rn >= ").append(startRow);
        return sql.toString();
    }

    /**
     * 构建排序子句
     */
    private String buildOrderBy(Paged paged) {
        // 优先使用sortByList进行多字段排序
        if (paged.getSortByList() != null && !paged.getSortByList().isEmpty()) {
            StringBuilder orderBy = new StringBuilder(" ORDER BY ");
            boolean first = true;

            for (String sortBy : paged.getSortByList()) {
                if (!first) {
                    orderBy.append(", ");
                }
                first = false;

                // 解析排序规则，格式如"field asc"或"field desc"
                String[] parts = sortBy.trim().split("\\s+");
                if (parts.length < 2) {
                    continue; // 跳过格式不正确的排序规则
                }

                String fieldName = parts[0];
                String direction = "DESC".equalsIgnoreCase(parts[1]) ? "DESC" : "ASC";

                // 处理字段名大小写问题：将驼峰命名转换为下划线大写（数据库字段格式）
                fieldName = camelCaseToUnderScoreUpperCase(fieldName).toUpperCase();

                orderBy.append(fieldName).append(" ").append(direction);
            }

            // 如果有有效的排序规则，返回构建的ORDER BY子句
            if (!first) {
                return orderBy.toString();
            }
        }

        // 使用原有sortBy和desc字段进行单字段排序
        if (paged.getSortBy() != null && !paged.getSortBy().isEmpty()) {
            String direction = Boolean.TRUE.equals(paged.getDesc()) ? "DESC" : "ASC";
            // 处理字段名大小写问题：将驼峰命名转换为下划线大写（数据库字段格式）
            String fieldName = camelCaseToUnderScoreUpperCase(paged.getSortBy()).toUpperCase();
            return " ORDER BY " + fieldName + " " + direction;
        }

        // 默认按ID降序
        return " ORDER BY ID DESC";
    }

    /**
     * 构建条件SQL
     */
    private String buildConditionSql(ConditionEntity condition, MapSqlParameterSource params, int paramIndex) {
        if (condition == null || condition.getFieldName() == null || condition.getFieldValue() == null) {
            return null;
        }

        String paramName = "param" + paramIndex;
        ConditionEnum conditionType = ConditionEnum.getByName(condition.getConditionalType());
        if (conditionType == null) {
            conditionType = ConditionEnum.Equal;
        }

        // 处理字段名大小写问题：将驼峰命名转换为下划线大写（数据库字段格式）
        String fieldName = camelCaseToUnderScoreUpperCase(condition.getFieldName()).toUpperCase();

        switch (conditionType) {
            case Equal:
                params.addValue(paramName, condition.getSafeValue());
                return fieldName + " = :" + paramName;
            case NoEqual:
                params.addValue(paramName, condition.getSafeValue());
                return fieldName + " != :" + paramName;
            case Like:
                params.addValue(paramName, "%" + condition.getSafeValue() + "%");
                return fieldName + " LIKE :" + paramName;
            case GreaterThan:
                params.addValue(paramName, condition.getSafeValue());
                return fieldName + " > :" + paramName;
            case GreaterThanOrEqual:
                params.addValue(paramName, condition.getSafeValue());
                return fieldName + " >= :" + paramName;
            case LessThan:
                params.addValue(paramName, condition.getSafeValue());
                return fieldName + " < :" + paramName;
            case LessThanOrEqual:
                params.addValue(paramName, condition.getSafeValue());
                return fieldName + " <= :" + paramName;
            case In:
                // IN查询需要特殊处理
                String[] inValues = condition.getSafeValue().split(",");
                List<String> paramNames = new ArrayList<>();
                for (int i = 0; i < inValues.length; i++) {
                    String inParamName = paramName + "_" + i;
                    params.addValue(inParamName, inValues[i].trim());
                    paramNames.add(":" + inParamName);
                }
                return fieldName + " IN (" + String.join(",", paramNames) + ")";
            case NotIn:
                String[] notInValues = condition.getSafeValue().split(",");
                List<String> notInParamNames = new ArrayList<>();
                for (int i = 0; i < notInValues.length; i++) {
                    String notInParamName = paramName + "_" + i;
                    params.addValue(notInParamName, notInValues[i].trim());
                    notInParamNames.add(":" + notInParamName);
                }
                return fieldName + " NOT IN (" + String.join(",", notInParamNames) + ")";
            case IsEmpty:
                return "(" + fieldName + " IS NULL OR " + fieldName + " = '')";
            case NotEmpty:
                return "(" + fieldName + " IS NOT NULL AND " + fieldName + " != '')";
            default:
                params.addValue(paramName, condition.getSafeValue());
                return fieldName + " = :" + paramName;
        }
    }

    /**
     * 驼峰命名转下划线大写：prodProduct -> PROD_PRODUCT
     */
    private String camelCaseToUnderScoreUpperCase(String camelCase) {
        if (camelCase == null || camelCase.isEmpty()) {
            return camelCase;
        }

        StringBuilder result = new StringBuilder();
        for (int i = 0; i < camelCase.length(); i++) {
            char c = camelCase.charAt(i);
            if (Character.isUpperCase(c) && i > 0) {
                result.append('_');
            }
            result.append(Character.toUpperCase(c));
        }
        return result.toString();
    }
}