package com.aps.service.common;

import com.aps.entity.common.ConditionEntity;
import com.aps.entity.common.ConditionEnum;
import com.aps.entity.common.EntityConfig;
import com.aps.entity.common.Paged;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.jdbc.core.BeanPropertyRowMapper;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.jdbc.core.namedparam.MapSqlParameterSource;
import org.springframework.jdbc.core.namedparam.NamedParameterJdbcTemplate;
import org.springframework.stereotype.Service;

import java.util.*;

@Service
public class DatabaseQueryService {

    @Autowired
    private JdbcTemplate jdbcTemplate;

    @Autowired
    private NamedParameterJdbcTemplate namedParameterJdbcTemplate;

    /**
     * 查询数据库数据（分页+条件）- 修复Oracle分页问题
     */
    public Map<String, Object> queryDatabaseDataWithConditions(EntityConfig config, Paged paged) {
        String tableName = config.getTableName();

        // 构建基础SQL和参数
        StringBuilder whereClause = new StringBuilder(" WHERE 1=1");
        MapSqlParameterSource params = new MapSqlParameterSource();

        // 处理条件
        int paramIndex = 1;
        if (paged.getConditions() != null && !paged.getConditions().isEmpty()) {
            for (ConditionEntity condition : paged.getConditions()) {
                String conditionSql = buildConditionSql(condition, params, paramIndex);
                if (conditionSql != null) {
                    whereClause.append(" AND ").append(conditionSql);
                    paramIndex++;
                }
            }
        }

        // 构建排序
        String orderBy = buildOrderBy(paged);

        // 分页参数
        int page = paged.getPageIndex() != null ? paged.getPageIndex() : 1;
        int size = paged.getPageSize() != null ? paged.getPageSize() : 10;

        // 先查询总数
        String countSql = "SELECT COUNT(*) FROM " + tableName + whereClause.toString();
        Integer total;
        try {
            if (params.getValues().isEmpty()) {
                total = jdbcTemplate.queryForObject(countSql, Integer.class);
            } else {
                total = namedParameterJdbcTemplate.queryForObject(countSql, params, Integer.class);
            }
        } catch (Exception e) {
            total = 0;
        }

        // 查询数据（使用Oracle分页语法）
        List<?> records = Collections.emptyList();
        if (total != null && total > 0) {
            String dataSql = buildOraclePaginationSql(tableName, whereClause.toString(), orderBy, page, size);
            try {
                if (config.getEntityClass() != null) {
                    // 使用实体类映射
                    if (params.getValues().isEmpty()) {
                        records = jdbcTemplate.query(dataSql, new BeanPropertyRowMapper<>(config.getEntityClass()));
                    } else {
                        records = namedParameterJdbcTemplate.query(dataSql, params, new BeanPropertyRowMapper<>(config.getEntityClass()));
                    }
                } else {
                    // 返回Map列表
                    if (params.getValues().isEmpty()) {
                        records = jdbcTemplate.queryForList(dataSql);
                    } else {
                        records = namedParameterJdbcTemplate.queryForList(dataSql, params);
                    }
                }
            } catch (Exception e) {
                throw new RuntimeException("查询数据失败: " + e.getMessage(), e);
            }
        }

        Map<String, Object> result = new HashMap<>();
        result.put("records", records);
        result.put("total", total != null ? total : 0);
        result.put("current", page);
        result.put("size", size);

        return result;
    }

    /**
     * 查询数据库数据列表（不分页）
     */
    public List<Object> queryDatabaseDataList(EntityConfig config, Paged paged) {
        String tableName = config.getTableName();

        StringBuilder whereClause = new StringBuilder(" WHERE 1=1");
        MapSqlParameterSource params = new MapSqlParameterSource();

        // 处理条件
        int paramIndex = 1;
        if (paged.getConditions() != null && !paged.getConditions().isEmpty()) {
            for (ConditionEntity condition : paged.getConditions()) {
                String conditionSql = buildConditionSql(condition, params, paramIndex);
                if (conditionSql != null) {
                    whereClause.append(" AND ").append(conditionSql);
                    paramIndex++;
                }
            }
        }

        // 构建排序
        String orderBy = buildOrderBy(paged);

        String sql = "SELECT * FROM " + tableName + whereClause.toString() + orderBy;

        try {
            if (config.getEntityClass() != null) {
                if (params.getValues().isEmpty()) {
                    return new ArrayList<>(jdbcTemplate.query(sql, new BeanPropertyRowMapper<>(config.getEntityClass())));
                } else {
                    return new ArrayList<>(namedParameterJdbcTemplate.query(sql, params, new BeanPropertyRowMapper<>(config.getEntityClass())));
                }
            } else {
                if (params.getValues().isEmpty()) {
                    return new ArrayList<>(jdbcTemplate.queryForList(sql));
                } else {
                    return new ArrayList<>(namedParameterJdbcTemplate.queryForList(sql, params));
                }
            }
        } catch (Exception e) {
            throw new RuntimeException("查询数据列表失败: " + e.getMessage(), e);
        }
    }

    /**
     * 根据ID查询单条数据
     */
    public Object queryDatabaseDataById(EntityConfig config, String id) {
        String tableName = config.getTableName();

        // 尝试不同的ID字段名（兼容大小写）
        String[] idFields = {"ID", "Id", "id"};

        for (String idField : idFields) {
            try {
                String sql = "SELECT * FROM " + tableName + " WHERE " + idField + " = ?";
                if (config.getEntityClass() != null) {
                    return jdbcTemplate.queryForObject(sql, new BeanPropertyRowMapper<>(config.getEntityClass()), id);
                } else {
                    return jdbcTemplate.queryForMap(sql, id);
                }
            } catch (Exception e) {
                // 继续尝试下一个字段名
            }
        }

        return null; // 找不到数据时返回null而不是抛出异常
    }

    /**
     * 构建Oracle分页SQL
     */
    private String buildOraclePaginationSql(String tableName, String whereClause, String orderBy, int page, int size) {
        int startRow = (page - 1) * size + 1;
        int endRow = page * size;

        StringBuilder sql = new StringBuilder();
        sql.append("SELECT * FROM (");
        sql.append("    SELECT a.*, ROWNUM rn FROM (");
        sql.append("        SELECT * FROM ").append(tableName).append(whereClause).append(orderBy);
        sql.append("    ) a WHERE ROWNUM <= ").append(endRow);
        sql.append(") WHERE rn >= ").append(startRow);
        return sql.toString();
    }

    /**
     * 构建排序子句
     */
    private String buildOrderBy(Paged paged) {
        if (paged.getSortBy() != null && !paged.getSortBy().isEmpty()) {
            String direction = Boolean.TRUE.equals(paged.getDesc()) ? "DESC" : "ASC";
            return " ORDER BY " + paged.getSortBy() + " " + direction;
        }
        return " ORDER BY ID DESC"; // 默认按ID降序
    }

    /**
     * 构建条件SQL
     */
    private String buildConditionSql(ConditionEntity condition, MapSqlParameterSource params, int paramIndex) {
        if (condition == null || condition.getFieldName() == null || condition.getFieldValue() == null) {
            return null;
        }

        String paramName = "param" + paramIndex;
        ConditionEnum conditionType = ConditionEnum.getByName(condition.getConditionalType());
        if (conditionType == null) {
            conditionType = ConditionEnum.Equal;
        }

        // 处理字段名大小写问题：将驼峰命名转换为下划线大写（数据库字段格式）
        String fieldName = camelCaseToUnderScoreUpperCase(condition.getFieldName()).toUpperCase();

        switch (conditionType) {
            case Equal:
                params.addValue(paramName, condition.getSafeValue());
                return fieldName + " = :" + paramName;
            case NoEqual:
                params.addValue(paramName, condition.getSafeValue());
                return fieldName + " != :" + paramName;
            case Like:
                params.addValue(paramName, "%" + condition.getSafeValue() + "%");
                return fieldName + " LIKE :" + paramName;
            case GreaterThan:
                params.addValue(paramName, condition.getSafeValue());
                return fieldName + " > :" + paramName;
            case GreaterThanOrEqual:
                params.addValue(paramName, condition.getSafeValue());
                return fieldName + " >= :" + paramName;
            case LessThan:
                params.addValue(paramName, condition.getSafeValue());
                return fieldName + " < :" + paramName;
            case LessThanOrEqual:
                params.addValue(paramName, condition.getSafeValue());
                return fieldName + " <= :" + paramName;
            case In:
                // IN查询需要特殊处理
                String[] inValues = condition.getSafeValue().split(",");
                List<String> paramNames = new ArrayList<>();
                for (int i = 0; i < inValues.length; i++) {
                    String inParamName = paramName + "_" + i;
                    params.addValue(inParamName, inValues[i].trim());
                    paramNames.add(":" + inParamName);
                }
                return fieldName + " IN (" + String.join(",", paramNames) + ")";
            case NotIn:
                String[] notInValues = condition.getSafeValue().split(",");
                List<String> notInParamNames = new ArrayList<>();
                for (int i = 0; i < notInValues.length; i++) {
                    String notInParamName = paramName + "_" + i;
                    params.addValue(notInParamName, notInValues[i].trim());
                    notInParamNames.add(":" + notInParamName);
                }
                return fieldName + " NOT IN (" + String.join(",", notInParamNames) + ")";
            case IsEmpty:
                return "(" + fieldName + " IS NULL OR " + fieldName + " = '')";
            case NotEmpty:
                return "(" + fieldName + " IS NOT NULL AND " + fieldName + " != '')";
            default:
                params.addValue(paramName, condition.getSafeValue());
                return fieldName + " = :" + paramName;
        }
    }
    
    /**
     * 驼峰命名转下划线大写：prodProduct -> PROD_PRODUCT
     */
    private String camelCaseToUnderScoreUpperCase(String camelCase) {
        if (camelCase == null || camelCase.isEmpty()) {
            return camelCase;
        }

        StringBuilder result = new StringBuilder();
        for (int i = 0; i < camelCase.length(); i++) {
            char c = camelCase.charAt(i);
            if (Character.isUpperCase(c) && i > 0) {
                result.append('_');
            }
            result.append(Character.toUpperCase(c));
        }
        return result.toString();
    }
}