package com.aps.service.Algorithm;

import com.aps.common.util.ProductionDeepCopyUtil;
import com.aps.entity.Algorithm.*;
import com.aps.entity.basic.*;
import com.aps.service.plan.MachineSchedulerService;

import java.time.LocalDateTime;
import java.time.temporal.ChronoUnit;
import java.util.*;
import java.util.stream.Collectors;
import java.util.concurrent.locks.ReentrantLock;

/**
 * 作者：佟礼
 * 时间：2025-11-24
 */
public class GeneticDecoder {
    private final Map<String, Chromosome> decodingCache = new HashMap<>();
    // 缓存大小限制
    private static final int MAX_CACHE_SIZE = 1000;
    // 线程安全锁：避免多线程下缓存操作冲突（可选，若单线程可移除）
    private final ReentrantLock cacheLock = new ReentrantLock();
    private LocalDateTime baseTime;
    private final List<Machine> machines;
    private final MachineSchedulerService machineScheduler;
    private final List<Order> orders;
    private int orderMaxID;
    private final List<Material> materials;
    public GeneticDecoder(LocalDateTime baseTime, List<Machine> machines, List<Order> orders,
                          List<Material> materials, MachineSchedulerService machineScheduler) {
        this.baseTime = baseTime;
        this.machines = machines;
        this.orders = orders;
        this.orders.forEach(t -> t.setSYQuantity(t.getQuantity()));
        this.materials = materials;
        this.machineScheduler = machineScheduler;
        this.orderMaxID = orders.stream().mapToInt(Order::getId).max().orElse(0);
    }
    public void decodeChromosomeWithCache(Chromosome chromosome, List<GlobalOperationInfo> globalOpList,
                       List<Entry> allOperations) {

        // 1. 创建缓存键
        String cacheKey = createCacheKey(chromosome);

        // 2. 尝试从缓存获取（加锁保证线程安全）
        cacheLock.lock();
        try {
            if (decodingCache.containsKey(cacheKey)) {
                // 缓存命中：复用缓存结果
                Chromosome cachedResult = decodingCache.get(cacheKey);
                // 赋值给染色体
                String ID=chromosome.getID();
                chromosome= ProductionDeepCopyUtil.deepCopy(cachedResult);
                chromosome.setID(ID);
                return;
            }
        } finally {
            cacheLock.unlock(); // 确保锁释放
        }
        // 3. 缓存未命中：执行解码逻辑
        decode(chromosome, globalOpList,
                 allOperations);

        // 4. 将解码结果存入缓存（加锁保证线程安全）
        cacheLock.lock();
        try {
            // 存储缓存：封装为 CacheValue 对象
            decodingCache.put(
                    cacheKey,
                    chromosome
            );

            // 5. 限制缓存大小（超过 1000 移除最早插入的键）
            if (decodingCache.size() > MAX_CACHE_SIZE) {

                Iterator<String> keyIterator = decodingCache.keySet().iterator();
                if (keyIterator.hasNext()) {
                    keyIterator.next(); // 获取首个键
                    keyIterator.remove(); // 移除首个键（对应 C# _decodingCache.Keys.First()）
                }
            }
        } finally {
            cacheLock.unlock();
        }
    }
    /**
     * 染色体解码为调度方案
     */
    public void decode(Chromosome chromosome, List<GlobalOperationInfo> globalOpList,
                       List<Entry> allOperations) {
        Map<Integer, GlobalOperationInfo> globalOpMap = globalOpList.stream()
                .collect(Collectors.toMap(GlobalOperationInfo::getGlobalOpId, g -> g));

        // 验证：MachineSelection长度必须与全局工序数一致
        if (chromosome.getMachineSelection().size() != globalOpList.size()) {
            throw new IllegalArgumentException(String.format(
                    "MachineSelection长度（%d）与全局工序数（%d）不匹配！",
                    chromosome.getMachineSelection().size(), globalOpList.size()));
        }

        // 步骤1：生成“工序→机器/加工时间”映射
        List<OpMachine> opMachineMap = new ArrayList<>();
        for (GlobalOperationInfo globalOp : globalOpList) {
            int globalOpId = globalOp.getGlobalOpId();
            Entry op = globalOp.getOp();
            int groupId = globalOp.getGroupId();
            int sequence = globalOp.getSequence();

            // 从MachineSelection中获取当前工序的机器选择顺序号
            int machineSeq = chromosome.getMachineSelection().get(globalOpId);
            List<MachineOption> optionalMachines = op.getMachineOptions();

            // 验证：机器顺序号必须在可选范围内
            if (machineSeq < 1 || machineSeq > optionalMachines.size()) {
                throw new IllegalStateException(String.format(
                        "全局工序%d（订单%d工序%d）的机器顺序号%d超出范围（1-%d）",
                        globalOpId, groupId, sequence, machineSeq, optionalMachines.size()));
            }

            // 获取选择的设备ID和加工时间
            MachineOption selectedMachine = optionalMachines.get(machineSeq - 1);
            OpMachine opMachine=new OpMachine();
            opMachine.setGroupId(groupId);
            opMachine.setSequence(sequence);
            opMachine.setMachineId(selectedMachine.getMachineId());
            opMachine.setProcessingTime(selectedMachine.getProcessingTime());
            opMachineMap.add(opMachine);

        }

        // 步骤2：按OperationSequencing顺序调度工序
        Map<Integer, Integer> orderProcessCounter = allOperations.stream()
                .collect(Collectors.groupingBy(Entry::getGroupId, Collectors.collectingAndThen(
                        Collectors.counting(), Long::intValue)))
                .entrySet().stream()
                .collect(Collectors.toMap(Map.Entry::getKey, e -> 0));

        Map<Integer, Integer> orderLastEndTime = allOperations.stream()
                .collect(Collectors.groupingBy(Entry::getGroupId))
                .keySet().stream()
                .collect(Collectors.toMap(k -> k, k -> 0));

        Map<Integer, String> machineState = chromosome.getMachines().stream()
                .collect(Collectors.toMap(Machine::getId, m -> ""));

        List<Entry> allScheduledOps = new ArrayList<>();

        for (int groupId : chromosome.getOperationSequencing()) {
            int scheduledCount = orderProcessCounter.get(groupId);
            List<Entry> orderOps = allOperations.stream()
                    .filter(t -> t.getGroupId() == groupId)
                    .collect(Collectors.toList());

            if (scheduledCount >= orderOps.size()) {
                throw new IllegalStateException(String.format(
                        "订单%d的工序已全部调度（共%d道），无需重复处理！",
                        groupId, orderOps.size()));
            }

            Entry currentOp = orderOps.get(scheduledCount);
            int opSequence = currentOp.getSequence();

            // 从映射表中获取机器和加工时间
            OpMachine  machineInfo=opMachineMap.stream()
                    .filter(m -> m.getGroupId() == groupId&&m.getSequence()==opSequence)
                    .findFirst()
                    .orElse(null);
            int machineId = machineInfo.getMachineId();
            int processTime = machineInfo.getProcessingTime();
            Machine targetMachine = chromosome.getMachines().stream()
                    .filter(m -> m.getId() == machineId)
                    .findFirst()
                    .orElse(null);

            int prevtime = 0;

            if (!currentOp.getPrevEntryIds().isEmpty()) {
                // 处理多个前工序
                for (int opid : currentOp.getPrevEntryIds()) {
                    List<GAScheduleResult> prevOperations = chromosome.getResult().stream()
                            .filter(t -> t.getGroupId() == groupId && t.getOperationId() == opid)
                            .collect(Collectors.toList());

                    for (GAScheduleResult prevOp : prevOperations) {
                        prevtime = Math.max(prevtime, prevOp.getEndTime());
                    }
                }
            }

            // 上个离散参数
            String lastDiscreteParameter = machineState.get(machineId);

            int bomtime = 0;
            prevtime = Math.max(prevtime, bomtime);

            Machine machine = chromosome.getMachines().stream()
                    .filter(m -> m.getId() == machineId)
                    .findFirst()
                    .orElse(null);

            int changeoverTime =0; //(lastDiscreteParameter.isEmpty() ||
                   // lastDiscreteParameter.equals(currentOp.getDiscreteParameter())) ? 0 : 0;

            int actualEndTime = processWithSingleMachine(currentOp, machine, processTime, prevtime, chromosome);

            orderProcessCounter.put(groupId, orderProcessCounter.get(groupId) + 1);
            orderLastEndTime.put(groupId, actualEndTime);
            machineState.put(machineId, currentOp.getDiscreteParameter());
        }

        // 步骤4：计算调度指标
        calculateScheduleResult(chromosome);
    }



    private int processWithSingleMachine(Entry operation, Machine machine, int processingTime,
                                         int prevtime, Chromosome chromosome) {
        int processingTimeTotal = processingTime * operation.getQuantity();

        MachineCalculator machineCalculator=new MachineCalculator(baseTime,machines,machineScheduler);
        List<ScheduleResultDetail> geneDetails = machineCalculator.getNextAvailableTime(machine, prevtime, -1,
                processingTimeTotal, chromosome.getResult(), false, true, true);

        int startTime = geneDetails.stream()
                .mapToInt(ScheduleResultDetail::getStartTime)
                .min()
                .orElse(0);
        int endTime = geneDetails.stream()
                .mapToInt(ScheduleResultDetail::getEndTime)
                .max()
                .orElse(0);

        GAScheduleResult result=new GAScheduleResult();
        result.setGroupId(operation.getGroupId());
        result.setOperationId(operation.getId());
        result.setMachineId(machine.getId());
        result.setQuantity(operation.getQuantity());
        result.setStartTime(startTime);
        result.setEndTime(endTime);
        result.setOneTime(processingTime);
        result.setProcessingTime(processingTimeTotal);
        result.setGeneDetails(geneDetails);
        chromosome.getResult().add(result);
        return endTime;
    }

    private void calculateScheduleResult(Chromosome chromosome) {
        // 1. 最早完工时间（最小化）
        double makespan = chromosome.getResult().stream()
                .mapToInt(GAScheduleResult::getEndTime)
                .max()
                .orElse(0);

        // 2. 交付期满足情况（最小化延迟）
        double tardiness = 0;

        Map<Integer, List<GAScheduleResult>> orderGroups = chromosome.getResult().stream()
                .collect(Collectors.groupingBy(GAScheduleResult::getGroupId));

        for (Map.Entry<Integer, List<GAScheduleResult>> group : orderGroups.entrySet()) {
            int groupId = group.getKey();
            int orderCompletion = group.getValue().stream()
                    .mapToInt(GAScheduleResult::getEndTime)
                    .max()
                    .orElse(0);

            Order order = orders.stream()
                    .filter(t -> t.getId() == groupId)
                    .findFirst()
                    .orElse(null);


            LocalDateTime completionTime =baseTime.plusMinutes(orderCompletion);
            // 修复：正确处理OffsetDateTime到LocalDateTime的转换
            LocalDateTime dueDateTime = order.getDueDate().toLocalDateTime();

            if (completionTime.isAfter(dueDateTime)) {
                // 计算延迟小时数（修复时间计算）
                long hours = ChronoUnit.HOURS.between(dueDateTime, completionTime);
                long minutes = ChronoUnit.MINUTES.between(dueDateTime, completionTime) % 60;
                tardiness += hours*60 + (double) minutes ;
            }

        }

        // 3. 最小总换型时间
        double totalSetupTime = calculateTotalSetupTime(chromosome);

        // 4. 最小化总流程时间
        double totalFlowTime = calculateTotalFlowTime(chromosome);

        // 5. 机器负载均衡
        double machineLoadBalance = calculateMachineLoadBalance(chromosome);

        // 存储各目标值
        chromosome.setMakespan(makespan);
        chromosome.setTotalFlowTime(totalFlowTime);
        chromosome.setTotalChangeoverTime(totalSetupTime);
        chromosome.setMachineLoadStd(machineLoadBalance);
        chromosome.setDelayTime(tardiness);

        // 加权求和作为适应度（权重可根据需求调整）
//        chromosome.setFitness(0.4 * (1.0 / (1 + makespan)) +
//                0.3 * (1.0 / (1 + tardiness)) +
//                0.1 * (1.0 / (1 + totalSetupTime)) +
//                0.1 * (1.0 / (1 + totalFlowTime)) +
//                0.1 * machineLoadBalance);
    }

    private double calculateTotalFlowTime(Chromosome chromosome) {
        return chromosome.getResult().stream()
                .mapToDouble(GAScheduleResult::getFlowTime)
                .sum();
    }



    // 计算总换型时间
    private double calculateTotalSetupTime(Chromosome chromosome) {
        return chromosome.getResult().stream()
                .mapToDouble(GAScheduleResult::getChangeoverTime)
                .sum();
    }

    // 计算机器负载均衡指标
    private double calculateMachineLoadBalance(Chromosome chromosome) {
        Map<Integer, Double> machineUtilization = new HashMap<>();
        int maxEndTime = chromosome.getResult().stream()
                .mapToInt(GAScheduleResult::getEndTime)
                .max()
                .orElse(0);

        if (maxEndTime == 0) return 0;

        for (Machine machine : chromosome.getMachines()) {
            List<GAScheduleResult> machineGenes = chromosome.getResult().stream()
                    .filter(g -> g.getMachineId() == machine.getId())
                    .collect(Collectors.toList());

            double busyTime = machineGenes.stream()
                    .mapToInt(g -> g.getEndTime() - g.getStartTime())
                    .sum();

            machineUtilization.put(machine.getId(), busyTime / maxEndTime);
        }

        double avgUtilization = machineUtilization.values().stream()
                .mapToDouble(Double::doubleValue)
                .average()
                .orElse(0);

        double variance = machineUtilization.values().stream()
                .mapToDouble(u -> Math.pow(u - avgUtilization, 2))
                .sum() / chromosome.getMachines().size();

        // 方差越小，负载越均衡
        return 1.0 / (1 + variance);
    }
    /**
     * 补全：创建缓存键（核心逻辑，需与原 C# CreateCacheKey 一致）
     * 思路：将染色体的核心特征（机器选择+工序排序）拼接为字符串，确保相同染色体生成相同键
     */
    private String createCacheKey(Chromosome chromosome) {

        // 拼接机器选择：用 "," 分隔（例：1,3,2,4）
        String machineStr = chromosome.getMachineStr();

        // 拼接工序排序：用 "|" 分隔（例：2|1|3|4）
        String operationStr = chromosome.getOperationStr();

        // 组合最终键（用 "_" 分隔两部分，避免冲突）
        return machineStr + "_" + operationStr;
    }
}
