package com.aps.demo;

import com.aps.entity.Algorithm.GAScheduleResult;
import com.aps.entity.DiscreteParameterDuration;
import com.aps.entity.RoutingDiscreteParam;
import com.aps.entity.basic.Entry;
import com.aps.entity.basic.Machine;
import com.aps.service.DiscreteParameterDurationService;
import org.junit.jupiter.api.Test;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.context.ApplicationContext;

import java.math.BigDecimal;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.List;

@SpringBootTest
public class DiscreteParameterDurationTest {

    // 使用依赖注入而不是直接new
    @Autowired
    private DiscreteParameterDurationService calculator;

    @Autowired
    private ApplicationContext applicationContext;

    @Test
    public void testChangeoverRules() {
        System.out.println("=== 离散参数换型时间计算测试 ===\n");

        // 1. 创建测试数据
        List<GAScheduleResult> existingGenes = createTestTasks();
        List<Entry> allOperations = createTestOperations();
        Machine machine = createMachine();
        List<DiscreteParameterDuration> parameterDurations = createParameterConfigs();

        // 2. 测试场景1：工作时间长规则
        testWorkingDurationRule(existingGenes, allOperations, machine, parameterDurations);

        // 3. 测试场景2：绝对工作时间长规则
        testAbsoluteDurationRule(existingGenes, allOperations, machine, parameterDurations);

        // 4. 测试场景3：工作最大间隔规则
        testMaxIntervalRule(existingGenes, allOperations, machine, parameterDurations);

        // 5. 测试场景4：工单数量规则
        testOrderCountRule(existingGenes, allOperations, machine, parameterDurations);

        // 6. 测试场景5：累计数量规则
        testAccumulatedQuantityRule(existingGenes, allOperations, machine, parameterDurations);

        // 7. 测试场景6：参数切换换型
        testParameterChangeoverRule(existingGenes, allOperations, machine, parameterDurations);
    }

    /**
     * 测试场景1：工作时间长规则
     * 规则：工作时间超过7200秒（2小时）需要换型
     */
    private void testWorkingDurationRule(List<GAScheduleResult> existingGenes,
                                         List<Entry> allOperations,
                                         Machine machine,
                                         List<DiscreteParameterDuration> parameterDurations) {
        System.out.println("=== 测试场景1：工作时间长规则 ===");

        // 创建一个新的任务D
        Entry taskD = createOperation("TaskD", "001");
        taskD.setId(4);
        taskD.setExecId("D");

        // 计算换型时间
        double changeoverTime = calculator.calculateChangeoverTime(
                existingGenes, taskD, machine, allOperations, 36000); // 10小时后

        System.out.println("场景描述：A(2h)+B(2h)+C(2h)累计6小时 > 2小时阈值");
        System.out.println("任务D需要的换型时间：" + changeoverTime + "分钟");
        System.out.println("预期结果：需要换型\n");
    }

    /**
     * 测试场景2：绝对工作时间长规则
     * 规则：绝对工作时间超过14400秒（4小时）需要换型
     */
    private void testAbsoluteDurationRule(List<GAScheduleResult> existingGenes,
                                          List<Entry> allOperations,
                                          Machine machine,
                                          List<DiscreteParameterDuration> parameterDurations) {
        System.out.println("=== 测试场景2：绝对工作时间长规则 ===");

        Entry taskD = createOperation("TaskD", "001");
        taskD.setId(4);
        taskD.setExecId("D");

        double changeoverTime = calculator.calculateChangeoverTime(
                existingGenes, taskD, machine, allOperations, 21600); // 6小时后

        System.out.println("场景描述：从A开始到现在6小时 > 4小时阈值");
        System.out.println("任务D需要的换型时间：" + changeoverTime + "分钟");
        System.out.println("预期结果：需要换型\n");
    }

    /**
     * 测试场景3：工作最大间隔规则
     * 规则：任务间隔超过7200秒（2小时）需要换型
     */
    private void testMaxIntervalRule(List<GAScheduleResult> existingGenes,
                                     List<Entry> allOperations,
                                     Machine machine,
                                     List<DiscreteParameterDuration> parameterDurations) {
        System.out.println("=== 测试场景3：工作最大间隔规则 ===");

        Entry taskD = createOperation("TaskD", "001");
        taskD.setId(4);
        taskD.setExecId("D");

        // 设置任务D的开始时间在C之后3小时
        double changeoverTime = calculator.calculateChangeoverTime(
                existingGenes, taskD, machine, allOperations, 28800); // 8小时后（C结束2小时后）

        System.out.println("场景描述：C在6h结束，D在8h开始，间隔2小时=阈值");
        System.out.println("任务D需要的换型时间：" + changeoverTime + "分钟");
        System.out.println("预期结果：不需要换型（刚好等于阈值）\n");
    }

    /**
     * 测试场景4：工单数量规则
     * 规则：连续生产3个任务需要换型
     */
    private void testOrderCountRule(List<GAScheduleResult> existingGenes,
                                    List<Entry> allOperations,
                                    Machine machine,
                                    List<DiscreteParameterDuration> parameterDurations) {
        System.out.println("=== 测试场景4：工单数量规则 ===");

        Entry taskD = createOperation("TaskD", "001");
        taskD.setId(4);
        taskD.setExecId("D");

        double changeoverTime = calculator.calculateChangeoverTime(
                existingGenes, taskD, machine, allOperations, 21600);

        System.out.println("场景描述：A(1)+B(2)+C(3)+D(4)=连续4个相同参数任务");
        System.out.println("任务D需要的换型时间：" + changeoverTime + "分钟");
        System.out.println("预期结果：需要换型\n");
    }

    /**
     * 测试场景5：累计数量规则
     * 规则：累计数量达到3000个需要换型
     */
    private void testAccumulatedQuantityRule(List<GAScheduleResult> existingGenes,
                                             List<Entry> allOperations,
                                             Machine machine,
                                             List<DiscreteParameterDuration> parameterDurations) {
        System.out.println("=== 测试场景5：累计数量规则 ===");

        Entry taskD = createOperation("TaskD", "001");
        taskD.setId(4);
        taskD.setExecId("D");
        // 修正：使用Double类型
        taskD.setQuantity(1000.0);

        double changeoverTime = calculator.calculateChangeoverTime(
                existingGenes, taskD, machine, allOperations, 21600);

        System.out.println("场景描述：A(1000)+B(1000)+C(8000)累计10000个 > 3000阈值");
        System.out.println("任务D需要的换型时间：" + changeoverTime + "分钟");
        System.out.println("预期结果：需要换型\n");
    }

    /**
     * 测试场景6：参数切换换型
     * 规则：参数001切换到002需要换型
     */
    private void testParameterChangeoverRule(List<GAScheduleResult> existingGenes,
                                             List<Entry> allOperations,
                                             Machine machine,
                                             List<DiscreteParameterDuration> parameterDurations) {
        System.out.println("=== 测试场景6：参数切换换型规则 ===");

        // 创建参数不同的任务D
        Entry taskD = createOperation("TaskD", "002"); // 参数从001变为002
        taskD.setId(4);
        taskD.setExecId("D");

        double changeoverTime = calculator.calculateChangeoverTime(
                existingGenes, taskD, machine, allOperations, 21600);

        System.out.println("场景描述：A/B/C使用参数001，D使用参数002");
        System.out.println("任务D需要的换型时间：" + changeoverTime + "分钟");
        System.out.println("预期结果：需要换型（参数变化）\n");
    }

    /**
     * 创建测试任务（基于您的JSON数据）
     */
    private List<GAScheduleResult> createTestTasks() {
        List<GAScheduleResult> tasks = new ArrayList<>();

        // 任务A：0-7200秒（2小时），数量1000
        tasks.add(createTask("A", 0, 7200, 1, 1000.0));

        // 任务B：7200-14400秒（2小时），数量1000
        tasks.add(createTask("B", 7200, 14400, 1, 1000.0));

        // 任务C：14400-21600秒（2小时），数量8000
        tasks.add(createTask("C", 14400, 21600, 1, 8000.0));

        return tasks;
    }

    /**
     * 创建测试工序
     */
    private List<Entry> createTestOperations() {
        List<Entry> operations = new ArrayList<>();

        // 任务A
        Entry taskA = createOperation("TaskA", "001");
        taskA.setId(1);
        taskA.setExecId("A");
        // 修正：使用Double类型
        taskA.setQuantity(1000.0);
        operations.add(taskA);

        // 任务B
        Entry taskB = createOperation("TaskB", "001");
        taskB.setId(2);
        taskB.setExecId("B");
        taskB.setQuantity(1000.0);
        operations.add(taskB);

        // 任务C
        Entry taskC = createOperation("TaskC", "001");
        taskC.setId(3);
        taskC.setExecId("C");
        taskC.setQuantity(8000.0);
        operations.add(taskC);

        return operations;
    }

    /**
     * 创建测试设备
     */
    private Machine createMachine() {
        Machine machine = new Machine();
        machine.setId(1L); // Machine的ID可能是Long类型
        machine.setName("设备A");
        return machine;
    }

    /**
     * 创建参数配置
     */
    private List<DiscreteParameterDuration> createParameterConfigs() {
        List<DiscreteParameterDuration> configs = new ArrayList<>();

        // 工作时间长规则：2小时=7200秒
        configs.add(createConfig("001", "WorkingDuration", null,
                new BigDecimal("7200.0"), new BigDecimal("30.0")));

        // 绝对工作时间长规则：4小时=14400秒
        configs.add(createConfig("001", "AbsoluteDuration", null,
                new BigDecimal("14400.0"), new BigDecimal("30.0")));

        // 工作最大间隔规则：2小时=7200秒
        configs.add(createConfig("001", "MaxInterval", null,
                new BigDecimal("7200.0"), new BigDecimal("30.0")));

        // 工单数量规则：3个任务
        configs.add(createConfig("001", "OrderCount", null,
                new BigDecimal("3.0"), new BigDecimal("30.0")));

        // 累计数量规则：3000个
        configs.add(createConfig("001", "AccumulatedQuantity", null,
                new BigDecimal("3000.0"), new BigDecimal("30.0")));

        // 参数切换换型：参数001切换到002需要30分钟
        configs.add(createConfig("001", null, "002",
                null, new BigDecimal("30.0")));

        return configs;
    }

    /**
     * 创建GAScheduleResult任务
     */
    private GAScheduleResult createTask(String execId, int startTime, int endTime,
                                        int machineId, Double quantity) {
        GAScheduleResult task = new GAScheduleResult();
        task.setExecId(execId);
        task.setStartTime(startTime);
        task.setEndTime(endTime);
        task.setMachineId(machineId);
        task.setChangeoverTime(0);
        // 注意：GAScheduleResult的quantity字段也需要是Double类型
        // 如果GAScheduleResult的quantity是BigDecimal，需要转换
        // task.setQuantity(new BigDecimal(quantity.toString()));
        return task;
    }

    /**
     * 创建Entry工序
     */
    private Entry createOperation(String name, String parameterId) {
        Entry entry = new Entry();


        // 添加离散参数
        List<RoutingDiscreteParam> discreteParams = new ArrayList<>();
        RoutingDiscreteParam param = new RoutingDiscreteParam();
        param.setGroupId("参数组一");
        param.setParameterId(parameterId);
        discreteParams.add(param);

        entry.setDiscreteParameter(discreteParams);
        return entry;
    }

    /**
     * 创建参数配置
     */
    private DiscreteParameterDuration createConfig(String parameterId, String exp1,
                                                   String exp2, BigDecimal measureduration,
                                                   BigDecimal duration) {
        DiscreteParameterDuration config = new DiscreteParameterDuration();
        config.setId(java.util.UUID.randomUUID().toString());
        config.setCreationtime(LocalDateTime.now());
        config.setCreatoruserid(1L);
        config.setLastmodificationtime(LocalDateTime.now());
        config.setLastmodifieruserid(1L);
        config.setIsdeleted(0L);

        config.setGroupid("参数组一");
        config.setGroupname("参数组一");
        config.setParameterid(parameterId);
        config.setParametername("参数" + parameterId);

        config.setExp1(exp1);
        config.setExp2(exp2);

        if (measureduration != null) {
            config.setMeasureduration(measureduration.longValue());
        }
        if (duration != null) {
            config.setDuration(duration);
        }

        config.setUnit("分钟");
        config.setUnitid(1L);
        config.setMeasureunit("秒");
        config.setMeasureunitid(2L);

        return config;
    }

    /**
     * 测试数据验证方法
     */
    @Test
    public void testDataValidation() {
        System.out.println("=== 测试数据验证 ===");

        List<DiscreteParameterDuration> configs = createParameterConfigs();
        printConfigInfo(configs);

        // 验证数据类型
        for (DiscreteParameterDuration config : configs) {
            System.out.printf("配置验证 - ID: %s, 参数ID: %s, 规则类型: %s%n",
                    config.getId(), config.getParameterid(), config.getExp1());
            System.out.printf("  阈值: %s %s, 换型时间: %s %s%n",
                    config.getMeasureduration(), config.getMeasureunit(),
                    config.getDuration(), config.getUnit());
            System.out.println("  数据类型验证通过 ✓");
        }
    }

    /**
     * 边界条件测试
     */
    @Test
    public void testBoundaryConditions() {
        System.out.println("=== 边界条件测试 ===");

        // 测试1：刚好达到阈值
        testBoundaryScenario("刚好达到阈值", 7200.0, 7200.0);

        // 测试2：刚好超过阈值
        testBoundaryScenario("刚好超过阈值", 7201.0, 7200.0);

        // 测试3：刚好低于阈值
        testBoundaryScenario("刚好低于阈值", 7199.0, 7200.0);
    }

    private void testBoundaryScenario(String scenario, Double actual, Double threshold) {
        System.out.printf("场景: %s, 实际值: %s, 阈值: %s%n", scenario, actual, threshold);

        if (actual >= threshold) {
            System.out.println("  结果: 需要换型 ✓");
        } else {
            System.out.println("  结果: 不需要换型 ✓");
        }
    }

    /**
     * 辅助方法：打印配置信息
     */
    private void printConfigInfo(List<DiscreteParameterDuration> configs) {
        System.out.println("=== 当前配置信息 ===");
        for (DiscreteParameterDuration config : configs) {
            System.out.printf("参数: %s, 规则: %s, 阈值: %s, 换型时间: %s%n",
                    config.getParameterid(),
                    config.getExp1(),
                    config.getMeasureduration() != null ? config.getMeasureduration() + config.getMeasureunit() : "N/A",
                    config.getDuration() != null ? config.getDuration() + config.getUnit() : "N/A");
        }
        System.out.println();
    }
}