package com.aps.service.common;

import cn.hutool.core.bean.BeanUtil;
import com.aps.common.util.R;
import com.aps.entity.Algorithm.Chromosome;
import com.aps.entity.ProdProcessExec;
import com.aps.entity.basic.Entry;
import com.aps.entity.basic.MachineOption;
import com.aps.entity.basic.Order;
import com.aps.entity.common.*;
import com.aps.service.plan.PlanResultService;
import com.aps.service.plan.SceneService;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.fasterxml.jackson.datatype.jsr310.JavaTimeModule;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.jdbc.core.namedparam.MapSqlParameterSource;
import org.springframework.jdbc.core.namedparam.NamedParameterJdbcTemplate;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.lang.reflect.Field;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.LocalDateTime;
import java.time.OffsetDateTime;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeParseException;
import java.util.*;
import java.util.stream.Collectors;

@Service
public class ChromosomeDataService {

    @Autowired
    private SceneService sceneService;

    @Autowired
    private DatabaseQueryService databaseQueryService;

    @Autowired
    private NamedParameterJdbcTemplate namedParameterJdbcTemplate;

    @Autowired
    private Map<String, EntityConfig> entityConfigMap;

    @Autowired
    private PlanResultService planResultService;
    /**
     * 根据场景ID和实体名称查询Chromosome中的数据
     */

     ObjectMapper objectMapper = new ObjectMapper()
            .registerModule(new JavaTimeModule())
            .configure(SerializationFeature.WRITE_DATES_AS_TIMESTAMPS, false);
//    public Object queryChromosomeData(String sceneId, String entityName) {
//        EntityConfig config = getEntityConfig(entityName);
//
//        if (config.getDataSource() == DataSourceType.FILE) {
//            return queryFileData(sceneId, config);
//        } else {
//            // 数据库实体不需要sceneId，创建空的Paged对象
//            Paged paged = new Paged();
//            List<Object> result = databaseQueryService.queryDatabaseDataList(config, paged);
//            return result;
//        }
//    }
    /**
     * 驼峰命名转下划线大写：prodProduct -> PROD_PRODUCT
     */
    private String camelCaseToUnderScoreUpperCase(String camelCase) {
        if (camelCase == null || camelCase.isEmpty()) {
            return camelCase;
        }

        StringBuilder result = new StringBuilder();

        for (int i = 0; i < camelCase.length(); i++) {
            char c = camelCase.charAt(i);

            // 遇到大写字母且不是第一个字符时，添加下划线
            if (Character.isUpperCase(c) && i > 0) {
                // 检查前一个字符是否是小写，避免连续大写的情况
                char prevChar = camelCase.charAt(i - 1);
                if (Character.isLowerCase(prevChar)) {
                    result.append('_');
                }
            }
            result.append(Character.toUpperCase(c));
        }

        return result.toString();
    }
    /**
     * 根据场景ID和实体名称查询Chromosome中的分页数据，带条件过滤
     */
    public Map<String, Object> queryChromosomeDataWithConditions(String sceneId, String entityName, Paged paged) {
        EntityConfig config = getEntityConfig(entityName);

        if (config.getDataSource() == DataSourceType.FILE) {
            return queryFileDataWithConditions(sceneId, config, paged);
        } else {
            // 数据库查询
            return databaseQueryService.queryDatabaseDataWithConditions(config, paged);
        }
    }

    /**
     * 查询文件数据列表（支持条件过滤）
     */
    public List<Object> queryChromosomeDataList(String sceneId, String entityName, Paged paged) {
        EntityConfig config = getEntityConfig(entityName);

        if (config.getDataSource() == DataSourceType.FILE) {
            // 特殊处理：当实体是MachineOption时，需要先获取Entry列表，再提取符合条件的Entry的MachineOption
            if ("machineoption".equalsIgnoreCase(config.getEntityName())) {
                // 先获取Entry列表（而不是MachineOption列表）
                EntityConfig entryConfig = new EntityConfig();
                entryConfig.setEntityName("entry");
                entryConfig.setDataSource(DataSourceType.FILE);
                entryConfig.setFieldName("allOperations");
                
                Object entryData = queryFileData(sceneId, entryConfig);
                
                if (entryData instanceof List) {
                    List<?> entryList = (List<?>) entryData;
                    List<MachineOption> resultMachineOptions = new ArrayList<>();
                    
                    // 过滤Entry列表，只保留与场景ID匹配的Entry
                    for (Object obj : entryList) {
                        if (obj instanceof Entry) {
                            Entry entry = (Entry) obj;
                            // 检查Entry的SceneId是否匹配
                            if (entry.getSceneId() != null && !entry.getSceneId().equals(sceneId)) {
                                continue; // 跳过不属于当前场景的Entry
                            }
                            
                            // 检查Entry是否满足其他条件
                            boolean matchesOtherConditions = true;
                            if (!CollectionUtils.isEmpty(paged.getConditions())) {
                                for (ConditionEntity condition : paged.getConditions()) {
                                    if (condition != null && !"sceneId".equalsIgnoreCase(condition.getFieldName())) {
                                        if (!matchesCondition(entry, condition.getFieldName(), condition.getFieldValue(), 
                                                ConditionEnum.getByName(condition.getConditionalType()))) {
                                            matchesOtherConditions = false;
                                            break;
                                        }
                                    }
                                }
                            }
                            
                            if (matchesOtherConditions && entry.getMachineOptions() != null) {
                                // 如果Entry满足所有条件，添加其MachineOptions
                                resultMachineOptions.addAll(entry.getMachineOptions());
                            }
                        }
                    }
                    
                    return (List<Object>) (List<?>) resultMachineOptions;
                } else {
                    return new ArrayList<>();
                }
            } else if ("order".equalsIgnoreCase(config.getEntityName())) {
                // 特殊处理：当实体是Order时，使用orderId字段进行查询
                Object data = queryFileData(sceneId, config);
                
                if (data instanceof List) {
                    List<?> orderList = (List<?>) data;
                    List<Order> resultOrders = new ArrayList<>();
                    
                    // 过滤Order列表，只保留与场景ID匹配的Order（如果Order中有SceneId的话）
                    for (Object obj : orderList) {
                        if (obj instanceof Order) {
                            Order order = (Order) obj;
                            // 检查Order是否满足条件
                            boolean matchesOtherConditions = true;
                            if (!CollectionUtils.isEmpty(paged.getConditions())) {
                                for (ConditionEntity condition : paged.getConditions()) {
                                    if (condition != null && !"sceneId".equalsIgnoreCase(condition.getFieldName())) {
                                        if (!matchesCondition(order, condition.getFieldName(), condition.getFieldValue(), 
                                                ConditionEnum.getByName(condition.getConditionalType()))) {
                                            matchesOtherConditions = false;
                                            break;
                                        }
                                    }
                                }
                            }
                            
                            if (matchesOtherConditions) {
                                resultOrders.add(order);
                            }
                        }
                    }
                    
                    return (List<Object>) (List<?>) resultOrders;
                } else {
                    return new ArrayList<>();
                }
            } else {
                Object data = queryFileData(sceneId, config);
                return applyConditionsToList(data, paged);
            }
        } else {
            // 数据库查询
            return databaseQueryService.queryDatabaseDataList(config, paged);
        }
    }

    /**
     * 根据ID查询单条数据
     */
    public Object queryChromosomeDataById(String sceneId, String entityName, String id) {
        EntityConfig config = getEntityConfig(entityName);

        if (config.getDataSource() == DataSourceType.FILE) {
            return queryFileDataById(sceneId, config, id);
        } else {
            // 数据库查询
            return databaseQueryService.queryDatabaseDataById(config, id);
        }
    }

    /**
     * 获取实体配置（智能默认）
     */
    private EntityConfig getEntityConfig(String entityName) {
        if (entityName == null || entityName.isEmpty()) {
            throw new RuntimeException("实体名称不能为空");
        }

        String key = entityName.toLowerCase();
        EntityConfig config = entityConfigMap.get(key);

        if (config == null) {
            // 特殊处理：当实体是ProdProcessExec时，映射到allOperations字段
            if ("prodprocessexec".equalsIgnoreCase(key)) {
                config = new EntityConfig();
                config.setEntityName(entityName);
                config.setDataSource(DataSourceType.FILE);
                config.setFieldName("allOperations");
            }
            // 特殊处理：当实体是MachineOption时，映射到allOperations字段
            else if ("machineoption".equalsIgnoreCase(key)) {
                config = new EntityConfig();
                config.setEntityName(entityName);
                config.setDataSource(DataSourceType.FILE);
                config.setFieldName("allOperations");
            }
            // 特殊处理：当实体是KpiMetrics时，映射到kpiMetrics字段
            else if ("kpimetrics".equalsIgnoreCase(key)) {
                config = new EntityConfig();
                config.setEntityName(entityName);
                config.setDataSource(DataSourceType.FILE);
                config.setFieldName("kpiMetrics");
            } else {
                // 自动创建数据库配置（默认行为）
                config = createDefaultDbConfig(entityName);
                System.out.println(config.getTableName()
                );
            }
            // 缓存这个自动创建的配置
            entityConfigMap.put(key, config);
        }

        return config;
    }
//





    private EntityConfig createDefaultDbConfig(String entityName) {
        EntityConfig config = new EntityConfig();
        config.setEntityName(entityName);
        config.setDataSource(DataSourceType.DATABASE);

        // 智能表名映射
        String tableName = mapEntityNameToTableName(entityName);

        System.out.println("智能表名映射：" + entityName + " -> " + tableName);
        config.setTableName(tableName);

        // 如果没有对应的实体类，设置为null，查询返回Map
        config.setEntityClass(null);

        return config;
    }

    /**
     * 将实体名映射为数据库表名
     * 规则：ProdProduct -> PROD_PRODUCT
     */
    private String mapEntityNameToTableName(String entityName) {
        if (entityName == null || entityName.isEmpty()) {
            return entityName;
        }

        // 处理常见的命名约定
        String[] commonPrefixes = {"Prod", "Sys", "Tbl", "Tab", "Biz"};
        String processedName = entityName;

        // 检查是否有常见前缀
        for (String prefix : commonPrefixes) {
            if (entityName.startsWith(prefix) && entityName.length() > prefix.length()) {
                // 保留前缀，如ProdProduct -> PROD_PRODUCT
                break;
            }
        }

        // 驼峰命名转下划线大写
        return camelCaseToUnderScoreUpperCase(processedName);
    }

    /**
     * 查询文件数据
     */
    private Object queryFileData(String sceneId, EntityConfig config) {
        Chromosome chromosome = sceneService.loadChromosomeFromFile(sceneId);
        if (chromosome == null) {
            return null;
        }

        try {
            String fieldName = config.getFieldName();
            
//            // 特殊处理：当实体是ProdProcessExec时，使用allOperations字段
//            if ("prodprocessexec".equalsIgnoreCase(config.getEntityName())) {
//                fieldName = "allOperations";
//            }
//
            // 特殊处理：当实体是MachineOption时，使用allOperations字段
            if ("machineoption".equalsIgnoreCase(config.getEntityName())) {
                fieldName = "allOperations";
            }
            
            Field field = Chromosome.class.getDeclaredField(fieldName);
            field.setAccessible(true);
            Object result = field.get(chromosome);
            
//            // 如果实体是ProdProcessExec，但数据源是Entry，则进行转换
//            if ("prodprocessexec".equalsIgnoreCase(config.getEntityName())) {
//                return convertEntryToProdProcessExec(result);
//            }
            
            // 如果实体是MachineOption，提取Entry中的MachineOption列表
            if ("machineoption".equalsIgnoreCase(config.getEntityName())) {
                return extractMachineOptionsFromEntries(result);
            }
            
            // 如果实体是KpiMetrics，直接返回结果
            if ("kpimetrics".equalsIgnoreCase(config.getEntityName())) {
                return result;
            }
            
            return result;
        } catch (NoSuchFieldException e) {
            throw new RuntimeException("Chromosome类中未找到字段: " + config.getFieldName(), e);
        } catch (IllegalAccessException e) {
            throw new RuntimeException("访问Chromosome字段失败: " + e.getMessage(), e);
        }
    }

    /**
     * 查询文件数据（分页+条件）
     */
    private Map<String, Object> queryFileDataWithConditions(String sceneId, EntityConfig config, Paged paged) {
        Object data;
        
        // 特殊处理：当实体是MachineOption时，需要先获取Entry列表，再提取符合条件的Entry的MachineOption
        if ("machineoption".equalsIgnoreCase(config.getEntityName())) {
            // 先获取Entry列表（而不是MachineOption列表）
            EntityConfig entryConfig = new EntityConfig();
            entryConfig.setEntityName("entry");
            entryConfig.setDataSource(DataSourceType.FILE);
            entryConfig.setFieldName("allOperations");
            
            Object entryData = queryFileData(sceneId, entryConfig);
            
            if (entryData instanceof List) {
                List<?> entryList = (List<?>) entryData;
                List<MachineOption> resultMachineOptions = new ArrayList<>();
                
                // 过滤Entry列表，只保留与场景ID匹配的Entry
                for (Object obj : entryList) {
                    if (obj instanceof Entry) {
                        Entry entry = (Entry) obj;
                        // 检查Entry的SceneId是否匹配
                        if (entry.getSceneId() != null && !entry.getSceneId().equals(sceneId)) {
                            continue; // 跳过不属于当前场景的Entry
                        }
                        
                        // 检查Entry是否满足其他条件
                        boolean matchesOtherConditions = true;
                        if (!CollectionUtils.isEmpty(paged.getConditions())) {
                            for (ConditionEntity condition : paged.getConditions()) {
                                if (condition != null && !"sceneId".equalsIgnoreCase(condition.getFieldName())) {
                                    if (!matchesCondition(entry, condition.getFieldName(), condition.getFieldValue(), 
                                            ConditionEnum.getByName(condition.getConditionalType()))) {
                                        matchesOtherConditions = false;
                                        break;
                                    }
                                }
                            }
                        }
                        
                        if (matchesOtherConditions && entry.getMachineOptions() != null) {
                            // 如果Entry满足所有条件，添加其MachineOptions
                            resultMachineOptions.addAll(entry.getMachineOptions());
                        }
                    }
                }
                
                data = resultMachineOptions;
            } else {
                data = new ArrayList<MachineOption>();
            }
        } else if ("order".equalsIgnoreCase(config.getEntityName())) {
            // 特殊处理：当实体是Order时，使用orderId字段进行查询
            Object orderData = queryFileData(sceneId, config);
            
            if (orderData instanceof List) {
                List<?> orderList = (List<?>) orderData;
                List<Order> resultOrders = new ArrayList<>();
                
                // 过滤Order列表，只保留满足条件的Order
                for (Object obj : orderList) {
                    if (obj instanceof Order) {
                        Order order = (Order) obj;
                        
                        // 检查Order是否满足条件
                        boolean matchesOtherConditions = true;
                        if (!CollectionUtils.isEmpty(paged.getConditions())) {
                            for (ConditionEntity condition : paged.getConditions()) {
                                if (condition != null && !"sceneId".equalsIgnoreCase(condition.getFieldName())) {
                                    if (!matchesCondition(order, condition.getFieldName(), condition.getFieldValue(), 
                                            ConditionEnum.getByName(condition.getConditionalType()))) {
                                        matchesOtherConditions = false;
                                        break;
                                    }
                                }
                            }
                        }
                        
                        if (matchesOtherConditions) {
                            resultOrders.add(order);
                        }
                    }
                }
                
                data = resultOrders;
            } else {
                data = new ArrayList<Order>();
            }
        } else {
            data = queryFileData(sceneId, config);
            
            if (data instanceof List) {
                List<?> dataList = (List<?>) data;
                
                // 应用条件过滤
                if (!CollectionUtils.isEmpty(paged.getConditions())) {
                    dataList = filterDataByConditions(dataList, paged.getConditions());
                }
                
                data = dataList;
            }
        }

        Map<String, Object> result = new HashMap<>();
        int page = paged.getPageIndex() != null ? paged.getPageIndex() : 1;
        int size = paged.getPageSize() != null ? paged.getPageSize() : 10;

        if (data instanceof List) {
            List<?> dataList = (List<?>) data;

            int total = dataList.size();
            int fromIndex = (page - 1) * size;
            fromIndex = Math.min(fromIndex, total);
            int toIndex = Math.min(fromIndex + size, total);

            List<?> pagedData = dataList.subList(fromIndex, toIndex);

            result.put("records", pagedData);
            result.put("totalCount", total);
            result.put("pageIndex", page);
            result.put("size", size);
        } else {
            result.put("records", data);
            result.put("totalCount", 1);
            result.put("pageIndex", 1);
            result.put("size", 1);
        }

        return result;
    }

    /**
     * 根据ID查询文件数据
     */
    private Object queryFileDataById(String sceneId, EntityConfig config, String id) {
        // 特殊处理：当实体是MachineOption时，需要先获取Entry列表，再提取特定Entry的MachineOption
        if ("machineoption".equalsIgnoreCase(config.getEntityName())) {
            // 先获取Entry列表（而不是MachineOption列表）
            EntityConfig entryConfig = new EntityConfig();
            entryConfig.setEntityName("entry");
            entryConfig.setDataSource(DataSourceType.FILE);
            entryConfig.setFieldName("allOperations");
            
            Object result = queryFileData(sceneId, entryConfig);
            
            if (result instanceof List) {
                List<?> entryList = (List<?>) result;
                // 查找ID匹配的Entry
                for (Object obj : entryList) {
                    if (obj instanceof Entry) {
                        Entry entry = (Entry) obj;
                        // 检查Entry的SceneId是否匹配（如果Entry中有SceneId的话）
                        if (entry.getSceneId() != null && !entry.getSceneId().equals(sceneId)) {
                            continue; // 跳过不属于当前场景的Entry
                        }
                        // 检查Entry的ID是否匹配
                        if (String.valueOf(entry.getId()).equals(id)) {
                            // 如果ID匹配，返回该Entry的MachineOptions
                            return entry.getMachineOptions() != null ? entry.getMachineOptions() : new ArrayList<>();
                        }
                    }
                }
            }
            return null; // 找不到数据时返回null而不是抛出异常
        }

        Object result = queryFileData(sceneId, config);

        // 如果结果是List类型，则根据ID查找特定项
        if (result instanceof List) {
            List<Object> dataList = (List<Object>) result;
            Object item = dataList.stream()
                    .filter(obj -> {
                        try {
                            // 尝试获取对象的id字段（兼容大小写）
                            Field idField = getFieldIgnoreCase(obj.getClass(), "id");
                            if (idField != null) {
                                idField.setAccessible(true);
                                Object itemId = idField.get(obj);
                                return itemId != null && itemId.toString().equals(id);
                            }
                            return false;
                        } catch (Exception e) {
                            return false;
                        }
                    })
                    .findFirst()
                    .orElse(null);

            if (item != null) {
                return item;
            } else {
                return null; // 找不到数据时返回null而不是抛出异常
            }
        } else {
            // 如果不是列表，直接返回结果
            return result;
        }
    }

    /**
     * 忽略大小写获取字段
     */
    private Field getFieldIgnoreCase(Class<?> clazz, String fieldName) {
        try {
            return clazz.getDeclaredField(fieldName);
        } catch (NoSuchFieldException e) {
            // 尝试大写
            try {
                return clazz.getDeclaredField(fieldName.toUpperCase());
            } catch (NoSuchFieldException e1) {
                // 尝试小写
                try {
                    return clazz.getDeclaredField(fieldName.toLowerCase());
                } catch (NoSuchFieldException e2) {
                    return null;
                }
            }
        }
    }

    /**
     * 将数据应用条件过滤并返回列表
     */
    private List<Object> applyConditionsToList(Object data, Paged paged) {
        // 检查是否是直接传入的EntityConfig（这种情况可能需要获取场景ID和实体类型）
        // 实际上这个方法是在queryChromosomeDataList中调用的，此时数据已经从文件中获取
        
        if (data instanceof List) {
            List<Object> dataList = (List<Object>) data;

            // 应用条件过滤
            if (!CollectionUtils.isEmpty(paged.getConditions())) {
                dataList = filterDataByConditions(dataList, paged.getConditions());
            }

            return dataList;
        } else {
            List<Object> resultList = new ArrayList<>();
            resultList.add(data);
            return resultList;
        }
    }

    /**
     * 根据条件过滤数据（文件数据使用）- 修复版本
     */
    private List<Object> filterDataByConditions(List<?> dataList, List<ConditionEntity> conditions) {
        if (CollectionUtils.isEmpty(conditions)) {
            return new ArrayList<>(dataList);
        }

        List<Object> filteredList = new ArrayList<>(dataList);

        for (ConditionEntity condition : conditions) {
            if (condition == null || condition.getFieldName() == null || condition.getFieldValue() == null) {
                continue;
            }

            final String fieldName = condition.getFieldName();
            final String fieldValue = condition.getFieldValue();
            ConditionEnum conditionType = ConditionEnum.getByName(condition.getConditionalType());
            if (conditionType == null) {
                conditionType = ConditionEnum.Equal;
            }
            final ConditionEnum finalConditionType = conditionType;

            // 特殊处理：sceneId条件应该在查询层面处理，这里跳过
            if ("sceneId".equalsIgnoreCase(fieldName)) {
                continue;
            }

            // 使用增强的for循环替代stream，避免lambda变量问题
            List<Object> tempList = new ArrayList<>();
            for (Object item : filteredList) {
                if (matchesCondition(item, fieldName, fieldValue, finalConditionType)) {
                    tempList.add(item);
                }
            }
            filteredList = tempList;
        }

        return filteredList;
    }

    /**
     * 判断单个对象是否匹配条件
     */
    private boolean matchesCondition(Object item, String fieldName, String fieldValue, ConditionEnum conditionType) {
        try {
            // 获取字段值
            Object actualValue = getFieldValue(item, fieldName);
            if (actualValue == null) {
                return conditionType == ConditionEnum.IsEmpty;
            }

            String actualValueStr = actualValue.toString();

            switch (conditionType) {
                case Equal:
                    return actualValueStr.equals(fieldValue);
                case NoEqual:
                    return !actualValueStr.equals(fieldValue);
                case Like:
                    return actualValueStr.contains(fieldValue);
                case GreaterThan:
                    return compareValues(actualValueStr, fieldValue) > 0;
                case GreaterThanOrEqual:
                    return compareValues(actualValueStr, fieldValue) >= 0;
                case LessThan:
                    return compareValues(actualValueStr, fieldValue) < 0;
                case LessThanOrEqual:
                    return compareValues(actualValueStr, fieldValue) <= 0;
                case In:
                    return Arrays.asList(fieldValue.split(",")).contains(actualValueStr);
                case NotIn:
                    return !Arrays.asList(fieldValue.split(",")).contains(actualValueStr);
                case IsEmpty:
                    return actualValueStr.isEmpty();
                case NotEmpty:
                    return !actualValueStr.isEmpty();
                default:
                    return actualValueStr.equals(fieldValue);
            }
        } catch (Exception e) {
            // 如果匹配过程中出错，保守起见返回true（不过滤掉数据）
            return true;
        }
    }

    /**
     * 安全获取字段值（支持多种字段名格式）
     */
    private Object getFieldValue(Object item, String fieldName) {
        // 尝试多种可能的字段名
        String[] possibleNames = {
                fieldName,
                fieldName.toLowerCase(),
                fieldName.toUpperCase(),
                toCamelCase(fieldName),
                toUnderScoreCase(fieldName)
        };

        for (String name : possibleNames) {
            try {
                Field field = item.getClass().getDeclaredField(name);
                field.setAccessible(true);
                Object value = field.get(item);
                if (value != null) {
                    return value;
                }
            } catch (Exception e) {
                // 继续尝试下一个名称
            }
        }

        return null;
    }

    /**
     * 比较值（支持数字和字符串）
     */
    private int compareValues(String str1, String str2) {
        try {
            // 尝试数字比较
            Double num1 = Double.parseDouble(str1);
            Double num2 = Double.parseDouble(str2);
            return num1.compareTo(num2);
        } catch (NumberFormatException e) {
            // 数字解析失败，使用字符串比较
            return str1.compareTo(str2);
        }
    }

    /**
     * 转换为驼峰命名
     */
    private String toCamelCase(String str) {
        if (str == null || str.isEmpty()) return str;
        if (str.length() == 1) return str.toLowerCase();
        return str.substring(0, 1).toLowerCase() + str.substring(1);
    }

    /**
     * 转换为下划线命名
     */
    private String toUnderScoreCase(String str) {
        if (str == null || str.isEmpty()) return str;
        StringBuilder result = new StringBuilder();
        for (int i = 0; i < str.length(); i++) {
            char c = str.charAt(i);
            if (Character.isUpperCase(c) && i > 0) {
                result.append('_');
            }
            result.append(Character.toLowerCase(c));
        }
        return result.toString();
    }
    /**
     * 更新Chromosome中的数据（支持文件实体和数据库实体）
     */
    public boolean updateChromosomeData(String sceneId, String entityName, Map<String, Object> data) {
        EntityConfig config = getEntityConfig(entityName);

        if (config.getDataSource() == DataSourceType.FILE) {
            if ("entry".equalsIgnoreCase(entityName)) {
                convertNestedObjects(data);
            }
            return updateFileEntity(sceneId, entityName, data);
        } else {
            // 数据库实体更新
            return updateDatabaseEntity(entityName, data);
        }
    }
    
    /**
     * 更新文件实体数据
     */
    private boolean updateFileEntity(String sceneId, String entityName, Map<String, Object> data) {
        EntityConfig config = getEntityConfig(entityName);

        // 从文件中加载Chromosome对象
        Chromosome chromosome = sceneService.loadChromosomeFromFile(sceneId);
        if (chromosome == null) {
            throw new RuntimeException("未找到场景ID为 " + sceneId + " 的Chromosome数据");
        }

        try {
            // 根据实体名称映射到Chromosome中的字段名
            String fieldName = config.getFieldName();

            // 使用反射设置字段值
            Field field = Chromosome.class.getDeclaredField(fieldName);
            field.setAccessible(true);

            // 获取原始数据
            Object originalData = field.get(chromosome);

            // 如果原始数据是List类型，并且传入的data包含id字段，则更新列表中的特定项
            if (originalData instanceof List && data.containsKey("id")) {
                List<Object> dataList = (List<Object>) originalData;
                String id = data.get("id").toString();

                // 查找并更新匹配的项
                for (int i = 0; i < dataList.size(); i++) {
                    Object item = dataList.get(i);
                    try {
                        Field idField = item.getClass().getDeclaredField("id");
                        idField.setAccessible(true);
                        Object itemId = idField.get(item);
                        if (itemId != null && itemId.toString().equals(id)) {
                            // 找到匹配项，更新它
                            updateObjectFields(item, data);
                            break;
                        }
                    } catch (Exception e) {
                        // 如果没有id字段或访问失败，尝试使用Id字段
                        try {
                            Field idField = item.getClass().getDeclaredField("Id");
                            idField.setAccessible(true);
                            Object itemId = idField.get(item);
                            if (itemId != null && itemId.toString().equals(id)) {
                                // 找到匹配项，更新它
                                updateObjectFields(item, data);
                                break;
                            }
                        } catch (Exception ex) {
                            // 忽略异常，继续下一个
                        }
                    }
                }
            } else {
                // 如果不是列表类型或没有id字段，抛出异常提示
                throw new RuntimeException("更新文件实体时必须提供id字段以标识要更新的具体对象");
            }

            if (entityName.equals("order")) {
                Order bean = BeanUtil.toBean(data, Order.class);
                Order order = chromosome.getOrders().stream().filter(o -> o.getId() == (bean.getId())).findFirst().orElse(null);
                chromosome = planResultService.editOrder(chromosome, sceneId, order);

            }else if (entityName.equals("entry")) {
                // 获取原有entry对象
                List<Object> dataList = (List<Object>) originalData;
                String id = data.get("id").toString();
                Entry originalEntry = null;
                for (Object item : dataList) {
                    if (item instanceof Entry) {
                        Entry entry = (Entry) item;
                        if (String.valueOf(entry.getId()).equals(id)) {
                            originalEntry = entry;
                            break;
                        }
                    }
                }

                // 如果找到了原有entry对象，直接使用它
                if (originalEntry != null) {
                    // 如果传入了machineOptions字段，需要特殊处理
                    if (data.containsKey("machineOptions")) {
                        List<?> machineOptionsList = (List<?>) data.get("machineOptions");
                        List<MachineOption> convertedOptions = new ArrayList<>();
                        for (Object item : machineOptionsList) {
                            // 手动转换每个MachineOption对象
                            MachineOption option = objectMapper.convertValue(item, MachineOption.class);
                            convertedOptions.add(option);
                        }
                        // 设置转换后的machineOptions
                        originalEntry.setMachineOptions(convertedOptions);
                        // 从data中移除machineOptions，避免updateObjectFields再次处理
                        data.remove("machineOptions");
                    }

                    // 更新其他传入的字段
                    updateObjectFields(originalEntry, data);
                    
                    // 调用editOperation方法
                    chromosome = planResultService.editOperation(chromosome, sceneId, originalEntry );
                    if (originalEntry.getDesignatedStartTime()== null){
                        planResultService.unlockStartTime(chromosome, originalEntry.getId());
                    }else {

                        List<Integer> opids = new ArrayList<>();
                        opids.add(originalEntry.getId());
                        chromosome = planResultService.lock(chromosome, opids, originalEntry.getDesignatedStartTime(), 0L, 1);

                    }

                }
                    // 其他字段通过updateObjectFields更新

            }

            // 保存更新后的Chromosome到文件
            return sceneService.saveChromosomeToFile(chromosome, sceneId);
        } catch (NoSuchFieldException e) {
            throw new RuntimeException("Chromosome类中未找到字段: " + entityName, e);
        } catch (IllegalAccessException e) {
            throw new RuntimeException("访问Chromosome字段失败: " + e.getMessage(), e);
        }
    }
    
    /**
     * 更新数据库实体数据
     */
    private boolean updateDatabaseEntity(String entityName, Map<String, Object> data) {
        EntityConfig config = getEntityConfig(entityName);
        String tableName = config.getTableName();
        
        if (data == null || !data.containsKey("id")) {
            throw new RuntimeException("更新数据库实体时必须提供id字段");
        }
        
        String id = data.get("id").toString();
        
        // 构建更新SQL
        StringBuilder sql = new StringBuilder();
        sql.append("UPDATE ").append(tableName).append(" SET ");
        
        List<String> setParts = new ArrayList<>();
        MapSqlParameterSource params = new MapSqlParameterSource();
        
        // 遍历要更新的字段，排除id字段
        for (Map.Entry<String, Object> entry : data.entrySet()) {
            String fieldName = entry.getKey();
            Object fieldValue = entry.getValue();
            
            if (!"id".equalsIgnoreCase(fieldName)) {
                // 将驼峰命名转换为下划线大写格式
                String dbFieldName = camelCaseToUnderScoreUpperCase(fieldName).toUpperCase();
                setParts.add(dbFieldName + " = :" + fieldName);
                params.addValue(fieldName, fieldValue);
            }
        }
        
        if (setParts.isEmpty()) {
            throw new RuntimeException("没有有效的字段用于更新");
        }
        
        sql.append(String.join(", ", setParts));
        
        // 添加WHERE条件
        sql.append(" WHERE ID = :idParam");
        params.addValue("idParam", id);
        
        try {
            int rowsAffected = namedParameterJdbcTemplate.update(sql.toString(), params);
            return rowsAffected > 0;
        } catch (Exception e) {
            throw new RuntimeException("数据库更新失败: " + e.getMessage(), e);
        }
    }

    /**
     * 批量操作Chromosome中的数据（仅支持文件实体）
     */
    public int batchChromosomeData(String sceneId, String entityName, List<Map<String, Object>> dataList) {
        EntityConfig config = getEntityConfig(entityName);

        if (config.getDataSource() == DataSourceType.DATABASE) {
            throw new RuntimeException("数据库实体暂不支持批量操作");
        }

        // 从文件中加载Chromosome对象
        Chromosome chromosome = sceneService.loadChromosomeFromFile(sceneId);
        if (chromosome == null) {
            throw new RuntimeException("未找到场景ID为 " + sceneId + " 的Chromosome数据");
        }

        try {
            // 根据实体名称映射到Chromosome中的字段名
            String fieldName = config.getFieldName();

            // 使用反射获取字段值
            Field field = Chromosome.class.getDeclaredField(fieldName);
            field.setAccessible(true);

            // 获取原始数据
            Object originalData = field.get(chromosome);

            // 如果原始数据是List类型，则批量更新列表中的项
            if (originalData instanceof List) {
                List<Object> originalList = (List<Object>) originalData;
                int successCount = 0;

                for (Map<String, Object> data : dataList) {
                    try {
                        if (data.containsKey("id")) {
                            String id = data.get("id").toString();

                            // 查找并更新匹配的项
                            for (Object item : originalList) {
                                try {
                                    Field idField = item.getClass().getDeclaredField("id");
                                    idField.setAccessible(true);
                                    Object itemId = idField.get(item);
                                    if (itemId != null && itemId.toString().equals(id)) {
                                        // 找到匹配项，更新它
                                        updateObjectFields(item, data);
                                        successCount++;
                                        break;
                                    }
                                } catch (Exception e) {
                                    // 如果没有id字段或访问失败，尝试使用Id字段
                                    try {
                                        Field idField = item.getClass().getDeclaredField("Id");
                                        idField.setAccessible(true);
                                        Object itemId = idField.get(item);
                                        if (itemId != null && itemId.toString().equals(id)) {
                                            // 找到匹配项，更新它
                                            updateObjectFields(item, data);
                                            successCount++;
                                            break;
                                        }
                                    } catch (Exception ex) {
                                        // 忽略异常，继续下一个
                                    }
                                }
                            }
                        }
                    } catch (Exception e) {
                        // 忽略单个数据的异常，继续处理下一个
                    }
                }

                // 保存更新后的Chromosome到文件
                boolean saved = sceneService.saveChromosomeToFile(chromosome, sceneId);
                return saved ? successCount : 0;
            } else {
                throw new RuntimeException("字段 " + entityName + " 不是列表类型，无法进行批量操作");
            }
        } catch (NoSuchFieldException e) {
            throw new RuntimeException("Chromosome类中未找到字段: " + entityName, e);
        } catch (IllegalAccessException e) {
            throw new RuntimeException("访问Chromosome字段失败: " + e.getMessage(), e);
        }
    }

    /**
     * 删除Chromosome中的数据（仅支持文件实体）
     */
    public int deleteChromosomeData(String sceneId, String entityName, List<Object> ids) {
        EntityConfig config = getEntityConfig(entityName);

        if (config.getDataSource() == DataSourceType.DATABASE) {
            throw new RuntimeException("数据库实体暂不支持删除操作");
        }

        // 从文件中加载Chromosome对象
        Chromosome chromosome = sceneService.loadChromosomeFromFile(sceneId);
        if (chromosome == null) {
            throw new RuntimeException("未找到场景ID为 " + sceneId + " 的Chromosome数据");
        }

        try {
            // 根据实体名称映射到Chromosome中的字段名
            String fieldName = config.getFieldName();

            // 使用反射获取字段值
            Field field = Chromosome.class.getDeclaredField(fieldName);
            field.setAccessible(true);

            // 获取原始数据
            Object originalData = field.get(chromosome);

            // 如果原始数据是List类型，则从列表中删除指定项
            if (originalData instanceof List) {
                List<Object> originalList = (List<Object>) originalData;
                int deleteCount = 0;

                // 创建ID字符串列表便于比较
                List<String> idStrings = ids.stream()
                        .map(Object::toString)
                        .collect(Collectors.toList());

                // 使用迭代器安全地删除元素
                Iterator<Object> iterator = originalList.iterator();
                while (iterator.hasNext()) {
                    Object item = iterator.next();
                    try {
                        Field idField = item.getClass().getDeclaredField("id");
                        idField.setAccessible(true);
                        Object itemId = idField.get(item);
                        if (itemId != null && idStrings.contains(itemId.toString())) {
                            // 找到匹配项，删除它
                            iterator.remove();
                            deleteCount++;
                        }
                    } catch (Exception e) {
                        // 如果没有id字段或访问失败，尝试使用Id字段
                        try {
                            Field idField = item.getClass().getDeclaredField("Id");
                            idField.setAccessible(true);
                            Object itemId = idField.get(item);
                            if (itemId != null && idStrings.contains(itemId.toString())) {
                                // 找到匹配项，删除它
                                iterator.remove();
                                deleteCount++;
                            }
                        } catch (Exception ex) {
                            // 忽略异常，继续下一个
                        }
                    }
                }

                // 保存更新后的Chromosome到文件
                boolean saved = sceneService.saveChromosomeToFile(chromosome, sceneId);
                return saved ? deleteCount : 0;
            } else {
                throw new RuntimeException("字段 " + entityName + " 不是列表类型，无法进行删除操作");
            }
        } catch (NoSuchFieldException e) {
            throw new RuntimeException("Chromosome类中未找到字段: " + entityName, e);
        } catch (IllegalAccessException e) {
            throw new RuntimeException("访问Chromosome字段失败: " + e.getMessage(), e);
        }
    }

    /**
     * 更新对象的字段值
     */
    private void updateObjectFields(Object obj, Map<String, Object> data) {
        Class<?> clazz = obj.getClass();
        for (Map.Entry<String, Object> entry : data.entrySet()) {
            String fieldName = entry.getKey();
            Object fieldValue = entry.getValue();

            try {
                // 尝试直接查找字段（区分大小写）
                Field field = null;
                try {
                    field = clazz.getDeclaredField(fieldName);
                } catch (NoSuchFieldException e) {
                    // 如果找不到，尝试忽略大小写查找
                    for (Field declaredField : clazz.getDeclaredFields()) {
                        if (declaredField.getName().equalsIgnoreCase(fieldName)) {
                            field = declaredField;
                            break;
                        }
                    }
                }
                
                if (field != null) {
                    field.setAccessible(true);

                    // 简单的类型转换处理
                    if (fieldValue != null) {
                        Class<?> fieldType = field.getType();
                        if (fieldType == String.class) {
                            field.set(obj, fieldValue.toString());
                        } else if (fieldType == Integer.class || fieldType == int.class) {
                            field.set(obj, Integer.parseInt(fieldValue.toString()));
                        } else if (fieldType == Long.class || fieldType == long.class) {
                            field.set(obj, Long.parseLong(fieldValue.toString()));
                        } else if (fieldType == Double.class || fieldType == double.class) {
                            field.set(obj, Double.parseDouble(fieldValue.toString()));
                        } else if (fieldType == Boolean.class || fieldType == boolean.class) {
                            field.set(obj, Boolean.parseBoolean(fieldValue.toString()));
                        } else if (fieldType == List.class) {
                            // 特殊处理List类型字段，特别是MachineOption列表
                            List<?> valueList = (List<?>) fieldValue;
                            if (!valueList.isEmpty() && valueList.get(0) instanceof Map) {
                                // 如果列表中的元素是Map（如JSON反序列化结果），需要特殊处理
                                // 对于MachineOption这类复杂对象，我们暂时直接赋值
                                field.set(obj, fieldValue);
                            } else {
                                field.set(obj, fieldValue);
                            }
                        } else if (fieldType == LocalDateTime.class) {
                            // 特殊处理LocalDateTime类型字段
                            if (fieldValue instanceof String) {
                                // 如果是字符串，尝试解析为LocalDateTime
                                String dateString = (String) fieldValue;
                                // 处理不同格式的日期时间字符串，包括带Z后缀的ISO格式
                                LocalDateTime parsedDateTime;
                                if (dateString.endsWith("Z")) {
                                    // 处理带Z后缀的格式，将其转换为带+00:00时区的格式
                                    dateString = dateString.substring(0, dateString.length() - 1) + "+00:00";
                                    parsedDateTime = OffsetDateTime.parse(dateString).toLocalDateTime();
                                } else {
                                    // 尝试解析为OffsetDateTime然后转换为LocalDateTime
                                    try {
                                        parsedDateTime = OffsetDateTime.parse(dateString).toLocalDateTime();
                                    } catch (DateTimeParseException e) {
                                        // 如果不是偏移格式，尝试直接解析为LocalDateTime
                                        parsedDateTime = LocalDateTime.parse(dateString, DateTimeFormatter.ISO_LOCAL_DATE_TIME);
                                    }
                                }
                                field.set(obj, parsedDateTime);
                            } else if (fieldValue instanceof LocalDateTime) {
                                field.set(obj, (LocalDateTime) fieldValue);
                            } else {
                                // 尝试将其他类型转换为字符串再解析
                                field.set(obj, LocalDateTime.parse(fieldValue.toString(), DateTimeFormatter.ISO_LOCAL_DATE_TIME));
                            }
                        } else {
                            field.set(obj, fieldValue);
                        }
                    } else {
                        field.set(obj, null);
                    }
                }
            } catch (Exception e) {
                // 忽略无法设置的字段
            }
        }
    }
    
    /**
     * 将Entry列表转换为ProdProcessExec列表
     */
    private Object convertEntryToProdProcessExec(Object data) {
        if (data instanceof List) {
            List<?> entryList = (List<?>) data;
            List<ProdProcessExec> result = new ArrayList<>();
            
            for (Object obj : entryList) {
                if (obj instanceof Entry) {
                    Entry entry = (Entry) obj;
                    ProdProcessExec prodProcessExec = convertEntryToProdProcessExecSingle(entry);
                    result.add(prodProcessExec);
                }
            }
            
            return result;
        } else if (data instanceof Entry) {
            Entry entry = (Entry) data;
            return convertEntryToProdProcessExecSingle(entry);
        }
        
        return data; // 如果不是Entry类型，直接返回原数据
    }
    
    /**
     * 将单个Entry转换为ProdProcessExec
     */
    private ProdProcessExec convertEntryToProdProcessExecSingle(Entry entry) {
        ProdProcessExec prodProcessExec = new ProdProcessExec();
        
        // 基本字段映射
        prodProcessExec.setExecId(entry.ExecId);
        prodProcessExec.setSceneId(entry.SceneId);
        prodProcessExec.setOrderId(entry.OrderId);

        prodProcessExec.setRoutingDetailId(entry.getRoutingDetailId());
        prodProcessExec.setTaskSeq(entry.getTaskSeq());
        prodProcessExec.setRoutingDetailName(entry.getRoutingDetailName());
        prodProcessExec.setMachineId(entry.getSelectMachineID());
        prodProcessExec.setEquipTypeName(entry.getEquipTypeName());
        prodProcessExec.setEquipTypeCode(entry.getEquipTypeCode());
//        prodProcessExec.setEquipTypeID(entry.getEquipTypeID());
        prodProcessExec.setRuntime(entry.getRuntime());
        prodProcessExec.setSingleOut(entry.getSingleOut());
        prodProcessExec.setPlanQty(entry.getQuantity());
        prodProcessExec.setLogicalOrder((long) entry.getSequence());
        
        // 设置ID字段
        prodProcessExec.setId(String.valueOf(entry.getId()));
        
        // 其他字段根据需要进行映射
        prodProcessExec.setCanInterrupt(entry.getIsInterrupt());
        prodProcessExec.setRoutingId(entry.getRoutingId());
        
        return prodProcessExec;
    }
    
    /**
     * 从Entry列表中提取MachineOption列表
     */
    private Object extractMachineOptionsFromEntries(Object data) {
        if (data instanceof List) {
            List<?> entryList = (List<?>) data;
            List<MachineOption> result = new ArrayList<>();
            
            
            for (Object obj : entryList) {
                if (obj instanceof Entry) {
                    Entry entry = (Entry) obj;
                    if (entry.getMachineOptions() != null) {
                        // 返回所有Entry中的MachineOption
                        result.addAll(entry.getMachineOptions());
                    } else {
                    }
                }
            }
            
            return result;
        } else if (data instanceof Entry) {
            Entry entry = (Entry) data;
            return entry.getMachineOptions() != null ? entry.getMachineOptions() : new ArrayList<>();
        }
        
        return new ArrayList<>(); // 如果不是Entry类型，返回空列表
    }

    private void convertNestedObjects(Map<String, Object> data) {
        // 转换machineOptions（支持大小写）
        convertListField(data, "MachineOptions", MachineOption.class);
        // 可以添加其他需要转换的字段
    }

    private <T> void convertListField(Map<String, Object> data, String fieldName, Class<T> targetClass) {
        if (data.containsKey(fieldName)) {
            Object fieldValue = data.get(fieldName);
            if (fieldValue instanceof List) {
                List<?> list = (List<?>) fieldValue;
                List<T> convertedList = new ArrayList<>();

                for (Object item : list) {
                    if (item instanceof LinkedHashMap) {
                        T convertedItem = BeanUtil.toBean((Map<?, ?>) item, targetClass);
                        convertedList.add(convertedItem);
                    }
                }

                data.put(fieldName, convertedList);
            }
        }
    }


    public R<String> updateMachineOption(String sceneId, String taskId, Map<String, Object> data) {

        // 1. 加载Chromosome对象
        Chromosome chromosome = sceneService.loadChromosomeFromFile(sceneId);


        // 2. 根据taskId找到对应的entry
        Entry targetEntry = null;
        for (Entry entry : chromosome.getAllOperations()) {
            if (String.valueOf(entry.getId()).equals(taskId)) {
                targetEntry = entry;
                break;
            }
        }

        Long machineId = null;
        Object machineIdObj = data.get("machineId");
        if (machineIdObj instanceof Long) {
            machineId = (Long) machineIdObj;
        } else if (machineIdObj instanceof Integer) {
            machineId = ((Integer) machineIdObj).longValue();
        } else if (machineIdObj instanceof String) {
            machineId = Long.parseLong((String) machineIdObj);
        }

        // 4. 在entry的machineOptions列表中找到对应的machineOption
        MachineOption targetMachineOption = null;
        for (MachineOption machineOption : targetEntry.getMachineOptions()) {
            if (machineOption.getMachineId().equals(machineId)) {
                targetMachineOption = machineOption;
                break;
            }
        }



        // 5. 只更新传入的字段，保留原有字段值
        updateMachineOptionFields(targetMachineOption, data);

        // 6. 调用editMachine方法
        planResultService.editMachine(chromosome, sceneId, targetEntry);

        // 7. 保存更新后的Chromosome
        boolean saved = sceneService.saveChromosomeToFile(chromosome, sceneId);

        return R.ok("更新成功");


    }

    /**
     * 只更新MachineOption的传入字段，保留原有字段值
     */
    private void updateMachineOptionFields(MachineOption machineOption, Map<String, Object> data) {
        // 处理runtime字段
        if (data.containsKey("runtime")) {
            Object runtimeObj = data.get("runtime");
            if (runtimeObj instanceof Integer) {
                machineOption.setRuntime(new BigDecimal((Integer) runtimeObj));
            } else if (runtimeObj instanceof Long) {
                machineOption.setRuntime(new BigDecimal((Long) runtimeObj));
            } else if (runtimeObj instanceof Double) {
                machineOption.setRuntime(new BigDecimal((Double) runtimeObj));
            } else if (runtimeObj instanceof String) {
                machineOption.setRuntime(new BigDecimal((String) runtimeObj));
            }
        }

        // 处理singleOut字段
        if (data.containsKey("singleOut")) {
            Object singleOutObj = data.get("singleOut");
            if (singleOutObj instanceof Integer) {
                machineOption.setSingleOut(new BigDecimal((Integer) singleOutObj));
            } else if (singleOutObj instanceof Long) {
                machineOption.setSingleOut(new BigDecimal((Long) singleOutObj));
            } else if (singleOutObj instanceof Double) {
                machineOption.setSingleOut(new BigDecimal((Double) singleOutObj));
            } else if (singleOutObj instanceof String) {
                machineOption.setSingleOut(new BigDecimal((String) singleOutObj));
            }
        }

        // 处理preTime字段
        if (data.containsKey("preTime")) {
            Object preTimeObj = data.get("preTime");
            if (preTimeObj instanceof Integer) {
                machineOption.setPreTime((Integer) preTimeObj);
            } else if (preTimeObj instanceof Long) {
                machineOption.setPreTime(((Long) preTimeObj).intValue());
            } else if (preTimeObj instanceof String) {
                machineOption.setPreTime(Integer.parseInt((String) preTimeObj));
            }
        }

        // 处理processingTime字段 - 修改为：时间(runtime) ÷ 数量(singleOut)
        if (data.containsKey("processingTime") || data.containsKey("singleOut") || data.containsKey("runtime")) {
            // 1. 获取singleOut值
            BigDecimal singleOut = machineOption.getSingleOut();
            if (data.containsKey("singleOut")) {
                Object singleOutObj = data.get("singleOut");
                if (singleOutObj instanceof Integer) {
                    singleOut = new BigDecimal((Integer) singleOutObj);
                } else if (singleOutObj instanceof Long) {
                    singleOut = new BigDecimal((Long) singleOutObj);
                } else if (singleOutObj instanceof Double) {
                    singleOut = new BigDecimal((Double) singleOutObj);
                } else if (singleOutObj instanceof String) {
                    singleOut = new BigDecimal((String) singleOutObj);
                } else if (singleOutObj instanceof BigDecimal) {
                    singleOut = (BigDecimal) singleOutObj;
                }
            }

            // 2. 获取runtime值
            BigDecimal runtime = machineOption.getRuntime();
            if (data.containsKey("runtime")) {
                Object runtimeObj = data.get("runtime");
                if (runtimeObj instanceof Integer) {
                    runtime = new BigDecimal((Integer) runtimeObj);
                } else if (runtimeObj instanceof Long) {
                    runtime = new BigDecimal((Long) runtimeObj);
                } else if (runtimeObj instanceof Double) {
                    runtime = new BigDecimal((Double) runtimeObj);
                } else if (runtimeObj instanceof String) {
                    runtime = new BigDecimal((String) runtimeObj);
                } else if (runtimeObj instanceof BigDecimal) {
                    runtime = (BigDecimal) runtimeObj;
                }
            }

            // 3. 计算processingTime = runtime ÷ singleOut（时间 ÷ 数量）
            double processingTime = 0.0;
            if (singleOut != null && runtime != null && runtime.compareTo(BigDecimal.ZERO) != 0) {
                // 考虑runtime单位转换：如果runtime是毫秒，转换为秒
                BigDecimal runtimeSeconds = runtime.divide(new BigDecimal(1000), 6, RoundingMode.HALF_UP);
                // 关键修改：时间（秒） ÷ 数量，保留2位小数
                processingTime = runtimeSeconds.divide(singleOut, 2, RoundingMode.HALF_UP).doubleValue();
            }

            // 4. 设置计算后的processingTime
            machineOption.setProcessingTime(processingTime);
        }


//
//        // 处理setupTime字段
//        if (data.containsKey("setupTime")) {
//            Object setupTimeObj = data.get("setupTime");
//            if (setupTimeObj instanceof Integer) {
//                machineOption.setSetupTime((Integer) setupTimeObj);
//            } else if (setupTimeObj instanceof Long) {
//                machineOption.setSetupTime(((Long) setupTimeObj).intValue());
//            } else if (setupTimeObj instanceof String) {
//                machineOption.setSetupTime(Integer.parseInt((String) setupTimeObj));
//            }
//        }
//
//        // 处理teardownTime字段
//        if (data.containsKey("teardownTime")) {
//            Object teardownTimeObj = data.get("teardownTime");
//            if (teardownTimeObj instanceof Integer) {
//                machineOption.setTeardownTime((Integer) teardownTimeObj);
//            } else if (teardownTimeObj instanceof Long) {
//                machineOption.setTeardownTime(((Long) teardownTimeObj).intValue());
//            } else if (teardownTimeObj instanceof String) {
//                machineOption.setTeardownTime(Integer.parseInt((String) teardownTimeObj));
//            }
//        }
//
//        // 处理contantTime字段
//        if (data.containsKey("contantTime")) {
//            Object contantTimeObj = data.get("contantTime");
//            if (contantTimeObj instanceof Integer) {
//                machineOption.setContantTime((Integer) contantTimeObj);
//            } else if (contantTimeObj instanceof Long) {
//                machineOption.setContantTime(((Long) contantTimeObj).intValue());
//            } else if (contantTimeObj instanceof String) {
//                machineOption.setContantTime(Integer.parseInt((String) contantTimeObj));
//            }
//        }

//        // 处理equipCode字段
//        if (data.containsKey("equipCode")) {
//            machineOption.setEquipCode(String.valueOf(data.get("equipCode")));
//        }
//
//        // 处理resourceCode字段
//        if (data.containsKey("resourceCode")) {
//            machineOption.setResourceCode(String.valueOf(data.get("resourceCode")));
//        }

        // 忽略id等未知字段
    }




}