package com.aps.service.Algorithm;

import com.aps.common.util.ProductionDeepCopyUtil;
import com.aps.entity.Algorithm.GAScheduleResult;
import com.aps.entity.Algorithm.ScheduleResultDetail;
import com.aps.entity.basic.*;
import com.aps.service.plan.AlgorithmScheduler8;
import com.aps.service.plan.MachineSchedulerService;
import com.baomidou.mybatisplus.core.toolkit.StringUtils;

import java.math.BigDecimal;
import java.time.Duration;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.time.temporal.ChronoUnit;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 作者：佟礼
 * 时间：2025-11-24
 */
public class MachineCalculator {
    private LocalDateTime baseTime;
    private final List<Machine> machines;
    private final MachineSchedulerService machineScheduler;

    public MachineCalculator(LocalDateTime baseTime, List<Machine> machines, MachineSchedulerService machineScheduler) {
        this.baseTime = baseTime;
        this.machines = machines;

        this.machineScheduler = machineScheduler;

    }



    /**
     * 获取机器下一个可用时间窗口（考虑班次约束）
     */
    public List<ScheduleResultDetail> getNextAvailableTime(Machine machine, int proposedStartTime,
                                                            int prevtime, int processingTime,
                                                            List<GAScheduleResult> existingTasks,
                                                            boolean isInterrupt, boolean istask,
                                                            double oneTime,double quantity,
                                                            boolean islockMachineTime) {
        LocalDateTime startTime = baseTime.plus(proposedStartTime, ChronoUnit.SECONDS);

        String prevtimestr = "";
        if (prevtime > -1) {
            prevtimestr = baseTime.plus(prevtime, ChronoUnit.SECONDS)
                    .format(DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss"));
        }

        // 查找合适的班次窗口
        return findEarliestStart(machine, processingTime, startTime, prevtimestr,
                existingTasks, oneTime, quantity, istask, islockMachineTime);
    }


    // 查找最早可用开始时间

    private List<ScheduleResultDetail> findEarliestStart(
            Machine machine, int processingTime, LocalDateTime currentTime,
            String prevtime, List<GAScheduleResult> existingTasks,double oneTime,double quantity, boolean checkprevtime, boolean islockMachineTime
    ) {
        List<GAScheduleResult> machineTasks = existingTasks.stream()
                .filter(t -> t.getMachineId() == machine.getId())
                .sorted(Comparator.comparingInt(GAScheduleResult::getStartTime))
                .collect(Collectors.toList());

        List<ScheduleResultDetail> times = new ArrayList<>();

        TimeSegment slot = GetCurrentOrNextShift(machine, currentTime, prevtime, checkprevtime);
        if (slot == null) return times;

        LocalDateTime prevTimeDateTime = StringUtils.isEmpty(prevtime) ? null : LocalDateTime.parse(prevtime);
        LocalDateTime startCandidate = slot.getStart().isAfter(prevTimeDateTime != null ? prevTimeDateTime : currentTime)
                ? slot.getStart()
                : (prevTimeDateTime != null ? prevTimeDateTime : currentTime);
        LocalDateTime endCandidate = startCandidate.plusSeconds(processingTime);

        if (endCandidate.isAfter(slot.getEnd())) {
            return CaldEarliestStart(machine, processingTime, currentTime, prevtime, machineTasks,oneTime,quantity, checkprevtime,islockMachineTime);
        } else {
            ScheduleResultDetail time = new ScheduleResultDetail();
            time.setKey(slot.getKey());
            time.setStartTime((int) ChronoUnit.SECONDS.between(baseTime, startCandidate));
            time.setEndTime((int) ChronoUnit.SECONDS.between(baseTime, endCandidate));
            time.setOneTime(oneTime);
            time.setQuantity(quantity);
            times.add(time);
            if(islockMachineTime) {
                RemoveMachineAvailable(machine, time);
            }
            return times;
        }
    }


    private List<ScheduleResultDetail> CaldEarliestStart(
            Machine machine, int processingTime, LocalDateTime currentTime,
            String prevtime, List<GAScheduleResult> machineTasks,double oneTime,double quantity, boolean checkprevtime, boolean islockMachineTime
    ) {
        int remainingTime = processingTime;
        LocalDateTime st = StringUtils.isEmpty(prevtime) ? currentTime : LocalDateTime.parse(prevtime);
        LocalDateTime prevEnd = LocalDateTime.of(2000, 1, 1, 0, 0, 0);
        List<ScheduleResultDetail> times = new ArrayList<>();
        List<ScheduleResultDetail> oldTimes = new ArrayList<>();

        while (remainingTime > 0) {
            TimeSegment shift = GetCurrentOrNextShift(machine, currentTime, prevtime, checkprevtime);
            if (shift == null) break;

            LocalDateTime shiftStart = shift.getStart();
            LocalDateTime shiftEnd = shift.getEnd();

            // 检查班次间冲突
            if (!prevEnd.isEqual(LocalDateTime.of(2000, 1, 1, 0, 0, 0))) {
                if (prevEnd.isBefore(currentTime)) {
                    // 检查班次间任务

                 boolean  hasTask=  CheckTask( machine, machineTasks, prevEnd, shiftStart);

                    if (hasTask) {
                        // 重置状态
                        currentTime = shiftStart;
                        st = shiftStart;
                        remainingTime = processingTime;
                        prevEnd = LocalDateTime.of(2000, 1, 1, 0, 0, 0);
                        oldTimes.addAll(ProductionDeepCopyUtil.deepCopyList(times));
                        times.clear();
                        continue;
                    }



                }
            }

            prevEnd = shiftEnd;
            // 计算有效时间
            LocalDateTime effectiveStart = st.isAfter(shiftStart) ? st : shiftStart;
            long availableSeconds = ChronoUnit.SECONDS.between(effectiveStart, shiftEnd);

            // 处理当前班次
            int processable = Math.min(remainingTime, (int) availableSeconds);
            remainingTime -= processable;
            currentTime = effectiveStart.plusSeconds(processable);

            // 添加时间详情
            ScheduleResultDetail time = new ScheduleResultDetail();
            time.setKey(shift.getKey());
            time.setStartTime((int) ChronoUnit.SECONDS.between(baseTime, effectiveStart));
            time.setEndTime((int) ChronoUnit.SECONDS.between(baseTime, currentTime));
            time.setQuantity((int)(processable/oneTime));
            time.setOneTime(oneTime);
            times.add(time);
            if (islockMachineTime) {
                // 还原未使用的时间段
                RemoveMachineAvailable(machine, time);
            }

        }

        // 还原未使用的时间段
        if (islockMachineTime) {
            // 还原未使用的时间段
            AddMachineAvailable(machine, oldTimes);
        }
        return times;
    }

    private boolean CheckTask(Machine machine,List<GAScheduleResult> machineTasks,LocalDateTime prevEnd,LocalDateTime shiftStart) {
        LocalDateTime finalPrevEnd = prevEnd;
        boolean hasTask = machineTasks.stream()
                .anyMatch(t -> {
                    LocalDateTime taskStart = baseTime.plusSeconds(t.getStartTime());
                    return taskStart.isAfter(finalPrevEnd) && taskStart.isBefore(shiftStart);
                });
        if (!hasTask) {
            // 检查班次间维修窗口
            if (machine.getMaintenanceWindows() != null) {
                LocalDateTime finalPrevEnd1 = prevEnd;
                boolean hasMaintenance = machine.getMaintenanceWindows().stream()
                        .anyMatch(w -> w.getStartTime().isAfter(finalPrevEnd1) && w.getStartTime().isBefore(shiftStart));

                return hasMaintenance;
            }
        }
        return hasTask;
    }

    /**
     * 获取设备当前或下一个有效班次
     */
    private TimeSegment GetCurrentOrNextShift(Machine machine, LocalDateTime time, String prevtime, boolean checkprevtime) {
        TimeSegment start = null;

        // 查找有效班次
        start = machine.getAvailability().stream()
                .filter(slot -> !slot.isUsed() && slot.getType() != SegmentType.MAINTENANCE)
                .filter(slot -> slot.getStart().isAfter(time) || slot.getEnd().isAfter(time))
                .findFirst()
                .orElse(null);

        if (start == null) {
            // 生成新时间段
            List<TimeSegment> timeSegments = machineScheduler.generateTimeSegment(machine, time.plusDays(1));
            machine.getAvailability().addAll(timeSegments);

            // 更新设备时间线
            Machine originalMachine = machines.stream()
                    .filter(t -> t.getId() == machine.getId())
                    .findFirst()
                    .orElse(null);
            if (originalMachine != null) {
                MachineTimeline timeline = machineScheduler.getOrCreateTimeline(originalMachine);
            }

            // 递归查找
            return GetCurrentOrNextShift(machine, time, prevtime, checkprevtime);
        }

        return start;
    }

    /**
     * 检查设备的可用开工时间
     * @param machine 目标设备
     * @param processingTime 加工时长（分钟）
     * @param currentTime 当前时间
     * @param startTime 开始时间
     * @param existingTasks 已有任务列表
     * @return 可用时间段列表，无可用时间返回null
     */
    public List<ScheduleResultDetail> checkMachineStartTime(Machine machine, int processingTime,
                                                   LocalDateTime currentTime, LocalDateTime startTime,
                                                   List<GAScheduleResult> existingTasks) {
        // 获取设备上已有任务并按开始时间排序
        List<GAScheduleResult> machineTasks = existingTasks.stream()
                .filter(t -> t.getMachineId() == machine.getId())
                .sorted(Comparator.comparing(GAScheduleResult::getStartTime))
                .collect(Collectors.toList());

        List<ScheduleResultDetail> times = new ArrayList<>();

        // 获取当前/上一个班次
        TimeSegment slot = getCurrentOrPrevShift(machine, currentTime);

        if (slot == null) {
            return null;
        }

        // 计算候选开始/结束时间（当前时间往前推加工时长）
        LocalDateTime startCandidate = currentTime.minusSeconds(processingTime);
        LocalDateTime endCandidate = currentTime;

        // 检查是否在可用时间段内
        if (startCandidate.isBefore(slot.getStart())) {
            // 放不下，寻找后续可用时间
            return checkMachineShift(machine, processingTime, currentTime, startTime, machineTasks);
        } else {
            // 生成可用时间段
            ScheduleResultDetail time = new ScheduleResultDetail();
            time.setKey(slot.getKey());
            // 计算相对基准时间的分钟数（对应原代码TotalMinutes
            time.setStartTime((int) ChronoUnit.SECONDS.between(baseTime, startCandidate));
            time.setEndTime((int) ChronoUnit.SECONDS.between(baseTime, endCandidate));
            times.add(time);
            return times;
        }
    }

    /**
     * 获取设备当前或上一个有效班次
     * @param machine 目标设备
     * @param time 目标时间
     * @return 有效班次，无则递归生成后返回
     */
    private TimeSegment getCurrentOrPrevShift(Machine machine, LocalDateTime time) {
        TimeSegment start = null;

        // 找到最后一个可用班次：未使用、非维护类型、开始时间早于目标时间
        // 倒序遍历找最后一个符合条件的班次
        List<TimeSegment> availability = machine.getAvailability();
        for (int i = availability.size() - 1; i >= 0; i--) {
            TimeSegment slot = availability.get(i);
            if (!slot.isUsed()
                    && slot.getType() != SegmentType.MAINTENANCE
                    && slot.getStart().isBefore(time)) {
                start = slot;
                break;
            }
        }

        // 若未找到班次，且目标时间晚于所有班次的结束时间 → 生成新班次并递归
        if (start == null) {
            // 获取所有班次的最大结束时间
            Optional<LocalDateTime> maxEndTime = availability.stream()
                    .map(TimeSegment::getEnd)
                    .max(LocalDateTime::compareTo);

            // 检查是否目标时间晚于最大结束时间（处理空列表情况）
            boolean isTimeAfterMaxEnd = maxEndTime.map(end -> time.isAfter(end)).orElse(true);
            if (isTimeAfterMaxEnd) {
                // 生成新的时间段并添加到设备可用班次中
                LocalDateTime generateStartTime = maxEndTime.orElse(LocalDateTime.now());
                List<TimeSegment> timeSegments = machineScheduler.generateTimeSegment(machine, generateStartTime);
                machine.getAvailability().addAll(timeSegments);

                // 更新机器列表中的对应机器
                Machine m = machines.stream()
                        .filter(t -> t.getId() == machine.getId())
                        .findFirst()
                        .orElse(null);
                if (m != null) {
                    MachineTimeline machineTimeline = machineScheduler.getOrCreateTimeline(m);
                    // 原代码注释：m.Availability = machineTimeline.Segments;
                    // 如需启用请补充：m.setAvailability(machineTimeline.getSegments());
                }

                // 递归调用，重新获取有效班次
                return getCurrentOrPrevShift(machine, time);
            }
        }

        return start;
    }
    /**
     * 检查设备班次，计算满足加工时长的可用时间段
     * @param machine 目标设备
     * @param processingTime 总加工时长（分钟）
     * @param currentTime 当前时间
     * @param startTime 开始时间阈值
     * @param machineTasks 设备已有任务列表
     * @return 分段的可用时间段列表，无可用时间返回null
     */
    private List<ScheduleResultDetail> checkMachineShift(Machine machine, int processingTime,
                                               LocalDateTime currentTime, LocalDateTime startTime,
                                               List<GAScheduleResult> machineTasks) {
        int remainingTime = processingTime;

        // 初始化上一班次结束时间（对应原2000-01-01的初始值）
        LocalDateTime prevEnd = LocalDateTime.of(2000, 1, 1, 0, 0);
        List<ScheduleResultDetail> times = new ArrayList<>();

        while (remainingTime > 0) {
            // 开始时间阈值超过当前时间，无可用时间
            if (startTime.isAfter(currentTime)) {
                return null;
            }

            // 获取当前/上一个有效班次
            TimeSegment shift = getCurrentOrPrevShift(machine, currentTime);
            if (shift == null) {
                return null;
            }
            LocalDateTime shiftStart = shift.getStart();
            LocalDateTime shiftEnd = shift.getEnd();

            // 非初始状态（上一班次已存在），检查班次连续性
            if (prevEnd.getYear()!=2000) {
                // 检查上一班次和当前班次之间是否有任务
                if (prevEnd.isAfter(shiftEnd)) {
                    // 查找任务：任务开始时间在当前班次开始 ~ 上一班次结束之间
                    LocalDateTime finalPrevEnd = prevEnd;
                    Optional<GAScheduleResult> task = machineTasks.stream()
                            .filter(t -> baseTime.plusSeconds(t.getStartTime()).isAfter(shiftStart)
                                    && baseTime.plusSeconds(t.getStartTime()).isBefore(finalPrevEnd))
                            .findFirst();

                    if (task.isPresent()) {
                        // 班次间有任务，重置状态重新查找
                        currentTime = shiftStart;
                        remainingTime = processingTime;
                        prevEnd = LocalDateTime.of(2000, 1, 1, 0, 0);
                        times.clear();
                        continue;
                    }
                }

                // 检查维护窗口
                if (machine.getMaintenanceWindows() != null && !machine.getMaintenanceWindows().isEmpty()) {
                    LocalDateTime finalPrevEnd = prevEnd;
                    Optional<MaintenanceWindow> maintenanceTask = machine.getMaintenanceWindows().stream()
                            .filter(t -> t.getStartTime().isAfter(shiftStart)
                                    && t.getStartTime().isBefore(finalPrevEnd))
                            .findFirst();

                    if (maintenanceTask.isPresent()) {
                        // 班次间有维护任务，重置状态重新查找
                        currentTime = shiftStart;
                        remainingTime = processingTime;
                        prevEnd = LocalDateTime.of(2000, 1, 1, 0, 0);
                        times.clear();
                        continue;
                    }
                }
            }

            // 调整班次有效结束时间（不超过当前时间）
            LocalDateTime effectiveEnd = shiftEnd.isAfter(currentTime) ? currentTime : shiftEnd;

            // 计算班次可用时长
            Duration availableDuration = Duration.between(shiftStart, effectiveEnd);
            long availableMinutes = availableDuration.getSeconds();

            // 当前班次可处理的时长（不超过剩余需要的时长）
            int processableMinutes = (int) Math.min(remainingTime, availableMinutes);

            // 计算当前班次的有效开始时间
            LocalDateTime effectiveStart = effectiveEnd.minusSeconds(processableMinutes);
            prevEnd = effectiveStart; // 更新上一班次结束时间
            remainingTime -= processableMinutes; // 剩余时长递减
            currentTime = effectiveStart; // 更新当前时间，继续循环

            // 生成当前班次的时间片段
            ScheduleResultDetail time = new ScheduleResultDetail();
            time.setKey(shift.getKey());


            time.setStartTime((int) ChronoUnit.SECONDS.between(baseTime, effectiveStart));
            time.setEndTime((int) ChronoUnit.SECONDS.between(baseTime, effectiveEnd));

            times.add(time);
        }

        return times;
    }

    private void RemoveMachineAvailable(Machine machine, ScheduleResultDetail geneDetails) {
        List<TimeSegment> timeSegments = new ArrayList<>();

        int index = machine.getAvailability().stream()
                .filter(t -> t.getKey().equals(geneDetails.getKey()))
                .findFirst()
                .map(machine.getAvailability()::indexOf)
                .orElse(-1);

        if (index > -1) {
            TimeSegment targetSegment = machine.getAvailability().get(index);
            LocalDateTime geneEndTime = baseTime.plusSeconds(geneDetails.getEndTime());

            if (targetSegment.getEnd().isAfter(geneEndTime)) {
                TimeSegment usedSegment = new TimeSegment();
                usedSegment.setStart(baseTime.plusSeconds(geneDetails.getStartTime()));
                usedSegment.setEnd(geneEndTime);
                usedSegment.setHoliday(false);
                usedSegment.setKey(UUID.randomUUID().toString());
                usedSegment.setType(SegmentType.REGULAR);
                usedSegment.setUsed(true);
                timeSegments.add(usedSegment);

                geneDetails.setKey(usedSegment.getKey());
                targetSegment.setStart(geneEndTime);
            } else {
                targetSegment.setUsed(true);
            }
        }

        if (!timeSegments.isEmpty()) {
            machine.getAvailability().addAll(timeSegments);
        }
        machine.getAvailability().sort(Comparator.comparing(TimeSegment::getStart));
    }

    public void AddMachineAvailable(Machine machine, List<ScheduleResultDetail> geneDetails) {
        if (geneDetails == null || geneDetails.isEmpty()) return;

        for (ScheduleResultDetail detail : geneDetails) {
            machine.getAvailability().stream()
                    .filter(t -> t.getKey().equals(detail.getKey()))
                    .findFirst()
                    .ifPresent(t -> t.setUsed(false));
        }

        machine.setAvailability(MergeSegments(machine.getAvailability()));
    }

    private List<TimeSegment> MergeSegments(List<TimeSegment> segments) {
        List<TimeSegment> maintenanceSegments = segments.stream()
                .filter(t -> t.getType() == SegmentType.MAINTENANCE)
                .collect(Collectors.toList());

        List<TimeSegment> unusedRegularSegments = segments.stream()
                .filter(t -> t.getType() != SegmentType.MAINTENANCE && !t.isUsed())
                .collect(Collectors.toList());

        List<TimeSegment> usedRegularSegments = segments.stream()
                .filter(t -> t.getType() != SegmentType.MAINTENANCE && t.isUsed())
                .collect(Collectors.toList());

        unusedRegularSegments.sort(Comparator.comparing(TimeSegment::getStart));

        Deque<TimeSegment> mergedUnused = new ArrayDeque<>();
        for (TimeSegment seg : unusedRegularSegments) {
            if (!mergedUnused.isEmpty() && mergedUnused.peekLast().getEnd().isAfter(seg.getStart())) {
                TimeSegment last = mergedUnused.pollLast();
                last.setEnd(last.getEnd().isAfter(seg.getEnd()) ? last.getEnd() : seg.getEnd());
                mergedUnused.offerLast(last);
            } else {
                mergedUnused.offerLast(seg);
            }
        }

        List<TimeSegment> result = new ArrayList<>(mergedUnused);
        result.addAll(usedRegularSegments);
        result.addAll(maintenanceSegments);
        result.sort(Comparator.comparing(TimeSegment::getStart));

        return result;
    }


}
