package com.aps.controller.common;

import com.aps.common.util.R;
import com.aps.entity.common.Paged;
import com.aps.service.common.ChromosomeDataService;
import io.swagger.v3.oas.annotations.tags.Tag;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import java.util.Collections;
import java.util.List;
import java.util.Map;

@RestController
@RequestMapping("/queryChromosome")
@Tag(name = "甘特图数据查询", description = "甘特图数据查询")
public class ChromosomeDataController {

    @Autowired
    private ChromosomeDataService chromosomeDataService;

    /**
     * 通用接口，根据实体名称查询Chromosome中的数据，支持分页和条件过滤
     * 示例:
     * - 文件实体: POST /queryChromosome/order/page
     *   Body: { "sceneId": "SCENE001", "pageIndex": 1, "pageSize": 10, "conditions": [...] }
     * - 数据库实体: POST /queryChromosome/user/page
     *   Body: { "pageIndex": 1, "pageSize": 10, "conditions": [...] }
     *
     * @param entityName 实体名称 (如: order, entry, machine, user, department等)
     * @param paged 分页和条件对象
     * @return 分页数据
     */
    @PostMapping("/{entityName}/page")
    public R<Map<String, Object>> queryChromosomeDataWithConditions(
            @PathVariable String entityName,
            @RequestBody Paged paged) {

        try {
            // 获取sceneId（文件实体需要，数据库实体可选）
            String sceneId = paged.getCondition("sceneId") != null ?
                    paged.getCondition("sceneId").getFieldValue() : null;

            // 文件实体必须要有sceneId
            if (isFileEntity(entityName) && (sceneId == null || sceneId.isEmpty())) {
                return R.failed("文件实体查询时sceneId不能为空");
            }

            Map<String, Object> result = chromosomeDataService.queryChromosomeDataWithConditions(
                    sceneId, entityName, paged);
            return R.ok(result);
        } catch (Exception e) {
            return R.failed("查询失败: " + e.getMessage());
        }
    }

    /**
     * 通用接口，根据实体名称查询Chromosome中的列表数据，支持条件过滤
     * 示例:
     * - 文件实体: POST /queryChromosome/order/list
     *   Body: { "sceneId": "SCENE001", "conditions": [...] }
     * - 数据库实体: POST /queryChromosome/user/list
     *   Body: { "conditions": [...] }
     *
     * @param entityName 实体名称 (如: order, entry, machine, user, department等)
     * @param paged 条件对象
     * @return 列表数据
     */
    @PostMapping("/{entityName}/list")
    public R<List<Object>> queryChromosomeDataList(
            @PathVariable String entityName,
            @RequestBody Paged paged) {

        try {
            // 获取sceneId（文件实体需要，数据库实体可选）
            String sceneId = paged.getCondition("sceneId") != null ?
                    paged.getCondition("sceneId").getFieldValue() : null;

            // 文件实体必须要有sceneId
            if (isFileEntity(entityName) && (sceneId == null || sceneId.isEmpty())) {
                return R.failed("文件实体查询时sceneId不能为空");
            }

            // 直接调用服务层的list查询方法
            List<Object> result = chromosomeDataService.queryChromosomeDataList(sceneId, entityName, paged);

            System.out.println("查询结果: " + result);
            if (result == null) {
                return R.ok(Collections.emptyList(), "查询成功，但未找到匹配的数据");
            }

            return R.ok(result);
        } catch (Exception e) {
            return R.failed("查询失败: " + e.getMessage());
        }
    }

    /**
     * 通用接口，根据实体名称和ID查询Chromosome中的单个数据
     * 示例:
     * - 文件实体: GET /queryChromosome/order/get?sceneId=xxx&id=123
     * - 数据库实体: GET /queryChromosome/user/get?id=123
     *
     * @param sceneId 场景ID (文件实体必需，数据库实体可选)
     * @param entityName 实体名称
     * @param id 数据ID
     * @return 单个数据对象
     */
    @GetMapping("/{entityName}/get")
    public R<Object> queryChromosomeDataById(
            @RequestParam(required = false) String sceneId,
            @PathVariable String entityName,
            @RequestParam String id) {

        try {
            // 文件实体必须要有sceneId
            if (isFileEntity(entityName) && (sceneId == null || sceneId.isEmpty())) {
                return R.failed("文件实体查询时sceneId不能为空");
            }

            Object result = chromosomeDataService.queryChromosomeDataById(sceneId, entityName, id);
            if (result != null) {
                return R.ok(result);
            } else {
                return R.failed("未找到ID为 " + id + " 的数据");
            }
        } catch (Exception e) {
            return R.failed("查询失败: " + e.getMessage());
        }
    }


    /**
     * 判断是否为文件实体
     */
    private boolean isFileEntity(String entityName) {
        // 这里列出所有文件实体的名称
        String[] fileEntities = {"order", "entry", "machine", "globaloperationinfo", "groupresult", "prodprocessexec"};
        for (String fileEntity : fileEntities) {
            if (fileEntity.equalsIgnoreCase(entityName)) {
                return true;
            }
        }
        return false;
    }
}