package com.aps.entity.Algorithm;

/**
 * 作者：佟礼
 * 时间：2025-12-26
 */
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.List;

// 时间区间实体类
public class DateRange {
    private LocalDateTime startDate;
    private LocalDateTime endDate;
    private static final DateTimeFormatter FORMATTER = DateTimeFormatter.ofPattern("yyyy/MM/dd HH:mm:ss");

    // 构造方法（支持字符串日期）
    public DateRange(String start, String end) {
        this.startDate = LocalDateTime.parse(start, FORMATTER);
        this.endDate = LocalDateTime.parse(end, FORMATTER);
    }

    // 构造方法（支持LocalDate）
    public DateRange(LocalDateTime start, LocalDateTime end) {
        this.startDate = start;
        this.endDate = end;
    }

    // 判断当前区间是否有效（start <= end）
    public boolean isValid() {
        return !startDate.isAfter(endDate);
    }

    // 切割当前区间，排除与targetRange重叠的部分，返回无交集的子区间列表
    public List<DateRange> subtract(DateRange targetRange) {
        List<DateRange> result = new ArrayList<>();
        // 1. 当前区间在targetRange之前：无交集，直接保留当前区间
        if (this.endDate.isBefore(targetRange.startDate)) {
            result.add(this);
            return result;
        }
        // 2. 当前区间在targetRange之后：无交集，直接保留当前区间
        if (this.startDate.isAfter(targetRange.endDate)) {
            result.add(this);
            return result;
        }
        // 3. 部分重叠：切割出无交集的子区间
        // 左半部分（当前start ~ target的start前一天）
        if (this.startDate.isBefore(targetRange.startDate)) {
            result.add(new DateRange(this.startDate, targetRange.startDate.minusSeconds(1)));
        }
        // 右半部分（target的end后一天 ~ 当前end）
        if (this.endDate.isAfter(targetRange.endDate)) {
            result.add(new DateRange(targetRange.endDate.plusSeconds(1), this.endDate));
        }
        return result;
    }

    // 转换为字符串格式（方便输出）
    @Override
    public String toString() {
        return startDate.format(FORMATTER) + " ~ " + endDate.format(FORMATTER);
    }

    // getter
    public LocalDateTime getStartDate() {
        return startDate;
    }

    public LocalDateTime getEndDate() {
        return endDate;
    }
}

// 核心方法实现

