package com.aps.service.Algorithm;

import com.aps.common.util.ProductionDeepCopyUtil;
import com.aps.entity.Algorithm.Chromosome;
import com.aps.entity.Algorithm.GlobalOperationInfo;
import com.aps.entity.Algorithm.OperationSequencingWeight;
import com.aps.entity.Algorithm.Pair;
import com.aps.entity.basic.Entry;
import com.aps.entity.basic.GlobalParam;
import com.aps.entity.basic.MachineOption;

import java.util.*;
import java.util.stream.Collectors;

/**
 * 作者：佟礼
 * 时间：2025-11-24
 */
public class GeneticOperations {
    private final Random rnd = new Random();
    private static List<Entry> allOperations;

    public GeneticOperations(List<Entry> allOperations) {
        GeneticOperations.allOperations = allOperations;
    }

    /**
     * 锦标赛选择
     */
    public List<Chromosome> tournamentSelection(List<Chromosome> population, int tournamentSize) {
        List<Chromosome> selected = new ArrayList<>();
        while (selected.size() < population.size()) {
            // 随机选择k个个体
//            List<Chromosome> candidates = population.stream()
//                    .sorted((a, b) -> rnd.nextInt())
//                    .limit(tournamentSize)
//                    .collect(Collectors.toList());
            // 步骤1: 创建一个临时列表用于存放候选者
            List<Chromosome> candidates = new ArrayList<>(population);
            // 步骤2: 随机打乱候选者列表
            Collections.shuffle(candidates, rnd);
            // 步骤3: 截取前 tournamentSize 个元素作为本次锦标赛的参与者
            if (tournamentSize < candidates.size()) {
                candidates = candidates.subList(0, tournamentSize);
            }
            // 步骤4: 从参与者中选择适应度最高的个体
            Chromosome best = candidates.stream()
                    .max(Comparator.comparingDouble(Chromosome::getFitness))
                    .orElseThrow(() -> new IllegalStateException("候选列表为空，无法选择最佳个体。"));
            // 复制个体（假设Chromosome有复制方法或通过构造函数复制）

            selected.add(ProductionDeepCopyUtil.deepCopy(best));
        }
        return selected;
    }

    // 重载，使用默认锦标赛大小3
    public List<Chromosome> tournamentSelection(List<Chromosome> population) {
        return tournamentSelection(population, 3);
    }

    /**
     * 改进POX交叉（工序排序部分）
     */
    public Pair<Chromosome, Chromosome> poxCrossover(Chromosome parent1, Chromosome parent2, int orderCount) {
        // 步骤1：随机划分工件集
        List<Integer> jobset1 = new ArrayList<>();
        for (int i = 0; i < orderCount; i++) {
            jobset1.add(i);
        }
        Collections.shuffle(jobset1, rnd);
        jobset1 = jobset1.subList(0, orderCount / 2);

        Set<Integer> jobset1Set = new HashSet<>(jobset1);
        List<Integer> jobset2 = new ArrayList<>();
        for (int i = 0; i < orderCount; i++) {
            if (!jobset1Set.contains(i)) {
                jobset2.add(i);
            }
        }

        // 步骤2：生成子代1
        List<Integer> child1Os = new ArrayList<>();
        for (int i = 0; i < parent1.getOperationSequencing().size(); i++) {
            int op1 = parent1.getOperationSequencing().get(i);
            int op2 = parent2.getOperationSequencing().get(i);
            if (jobset1Set.contains(op1) || jobset1Set.contains(op2)) {
                child1Os.add(op1);
            } else {
                child1Os.add(op2);
            }
        }

        // 步骤3：生成子代2
        List<Integer> child2Os = new ArrayList<>();
        for (int i = 0; i < parent2.getOperationSequencing().size(); i++) {
            int op1 = parent1.getOperationSequencing().get(i);
            int op2 = parent2.getOperationSequencing().get(i);
            if (jobset1Set.contains(op2) || jobset1Set.contains(op1)) {
                child2Os.add(op2);
            } else {
                child2Os.add(op1);
            }
        }

        // 机器选择部分交叉
        List<Integer> child1Ms = new ArrayList<>();
        List<Integer> child2Ms = new ArrayList<>();
        for (int i = 0; i < parent1.getMachineSelection().size(); i++) {
            int m1 = parent1.getMachineSelection().get(i);
            int m2 = parent2.getMachineSelection().get(i);
            if (rnd.nextDouble() < 0.5) {
                child1Ms.add(m1);
                child2Ms.add(m2);
            } else {
                child1Ms.add(m2);
                child2Ms.add(m1);
            }
        }

        Chromosome child1 = new Chromosome();

        child1.setMachineSelection(child1Ms);
        child1.setOperationSequencing(child1Os);

        Chromosome child2 = new Chromosome();

        child2.setMachineSelection(child2Ms);
        child2.setOperationSequencing(child2Os);

        return new Pair<>(child1, child2);
    }

    /**
     * 变异操作
     */
    public void mutate(Chromosome chromosome, List<GlobalOperationInfo> globalOpList, double baseMutationProb) {
        // 1. 机器选择部分变异
        mutateMachineSelection(chromosome, globalOpList, baseMutationProb);

        // 2. 工序排序部分变异
        mutateOperationSequencing(chromosome);
    }

    // 重载，使用默认变异概率
    public void mutate(Chromosome chromosome, List<GlobalOperationInfo> globalOpList) {
        mutate(chromosome, globalOpList, 0.1);
    }

    private void mutateMachineSelection(Chromosome chromosome, List<GlobalOperationInfo> globalOpList, double baseMutationProb) {
        // 计算变异位置数量
        int r = Math.max(1, (int) (chromosome.getMachineSelection().size() * baseMutationProb));
        int i = 0;
        while (i < r) {
            int pos = rnd.nextInt(chromosome.getMachineSelection().size());
            // 获取对应工序
            GlobalOperationInfo opInfo = globalOpList.get(pos);
            Entry operation = opInfo.getOp();
            int machineSeq = chromosome.getMachineSelection().get(pos);

            if (operation.getMachineOptions().size() == 1) {
                continue;
            } else {
                // 选择当前所选设备外最短加工时间的机器
                List<MachineOption> optionalMachines = operation.getMachineOptions();
                // 找到当前选中的机器
                MachineOption currentMachine = optionalMachines.get(machineSeq - 1);
                // 筛选其他机器并找到加工时间最短的
                MachineOption minLoadMachine = optionalMachines.stream()
                        .filter(m -> m.getMachineId() != currentMachine.getMachineId())
                        .min(Comparator.comparingDouble(m -> m.getProcessingTime()))
                        .orElse(currentMachine); // 如果没有其他机器，保持当前

                machineSeq = optionalMachines.indexOf(minLoadMachine) + 1;
                chromosome.getMachineSelection().set(pos, machineSeq);
                i++;
            }
        }
    }

    /**
     * 低优先级优先的工序排序变异
     */
    private void mutateOperationSequencing(Chromosome chromosome) {
        Random rnd = new Random();

        // 选择变异方式（交换或反转）
        if (rnd.nextDouble() < 0.5) {
            // 交换：优先选择高优先级工序的索引
            OperationSequencingWeight os1 = selectHighPriorityIndex(chromosome.getOperationSequencing(), 0);
            int idx1 = os1.getIndex();
            OperationSequencingWeight os2 = selectHighPriorityIndex(chromosome.getOperationSequencing(), os1.getWeight());
            int idx2 = os2.getIndex();

            // 交换位置
            List<Integer> os = chromosome.getOperationSequencing();
            Collections.swap(os, idx1, idx2);
        } else {
            // 反转：仅对高优先级工序集中的子序列反转
            OperationSequencingWeight osStart = selectHighPriorityIndex(chromosome.getOperationSequencing(), 0);
            int start = osStart.getIndex();
            OperationSequencingWeight osEnd = selectHighPriorityIndex(chromosome.getOperationSequencing(), osStart.getWeight());
            int end = osEnd.getIndex();

            if (start > end) {
                int temp = start;
                start = end;
                end = temp;
            }

            // 执行反转
            List<Integer> os = chromosome.getOperationSequencing();
            for (int i = 0; i < (end - start + 1) / 2; i++) {
                int pos1 = start + i;
                int pos2 = end - i;
                Collections.swap(os, pos1, pos2);
            }
        }
    }

    /**
     * 优先选择高优先级工序的索引（用于变异）
     */
    private OperationSequencingWeight selectHighPriorityIndex(List<Integer> os, int weight) {
        Random rnd = new Random();
        List<OperationSequencingWeight> indexWeights = CommonCalculator.getOsw(os, allOperations);

        if (!GlobalParam.IsBreakPriority) {
            if (weight == 0) {
                return indexWeights.get(rnd.nextInt(indexWeights.size()));
            } else {
                List<OperationSequencingWeight> filtered = indexWeights.stream()
                        .filter(t -> t.getWeight() == weight)
                        .collect(Collectors.toList());
                return filtered.get(rnd.nextInt(filtered.size()));
            }
        }

        return indexWeights.get(CommonCalculator.getOsIndex(indexWeights));
    }


}
