package com.aps.controller.common;

import com.aps.common.util.R;
import com.aps.entity.common.Paged;
import com.aps.service.common.ChromosomeDataService;
import io.swagger.v3.oas.annotations.tags.Tag;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import java.util.Collections;
import java.util.List;
import java.util.Map;

@RestController
@RequestMapping("/queryChromosome")
@Tag(name = "染色体数据查询", description = "根据实体名称查询染色体中的数据")
public class ChromosomeDataController {

    @Autowired
    private ChromosomeDataService chromosomeDataService;


    /**
     * 通用接口，根据实体名称查询Chromosome中的数据，支持分页和条件过滤
     * 示例: POST /queryChromosome/order/page 
     * Body: { "sceneId": "SCENE001", "pageIndex": 1, "pageSize": 10, "conditions": [...] }
     *
     * @param entityName 实体名称 (如: order, entry, machine等)
     * @param paged 分页和条件对象
     * @return 分页数据
     */
    @PostMapping("/{entityName}/page")
    public R<Map<String, Object>> queryChromosomeDataWithConditions(
            @PathVariable String entityName,
            @RequestBody Paged paged) {
        
        String sceneId = paged.getCondition("sceneId") != null ? 
            paged.getCondition("sceneId").getFieldValue() : null;
            
        if (sceneId == null || sceneId.isEmpty()) {
            return R.failed("sceneId不能为空");
        }
        
        Map<String, Object> result = chromosomeDataService.queryChromosomeDataWithConditions(
                sceneId, entityName, paged);
        return R.ok(result);
    }
    
    /**
     * 通用接口，根据实体名称查询Chromosome中的列表数据，支持条件过滤
     * 示例: POST /queryChromosome/order/list 
     * Body: { "sceneId": "SCENE001", "conditions": [...] }
     *
     * @param entityName 实体名称 (如: order, entry, machine等)
     * @param paged 条件对象
     * @return 列表数据
     */
    @PostMapping("/{entityName}/list")
    public R<List<Object>> queryChromosomeDataList(
            @PathVariable String entityName,
            @RequestBody Paged paged) {
        
        String sceneId = paged.getCondition("sceneId") != null ? 
            paged.getCondition("sceneId").getFieldValue() : null;
            
        if (sceneId == null || sceneId.isEmpty()) {
            return R.failed("sceneId不能为空");
        }
        
        Object result = chromosomeDataService.queryChromosomeData(sceneId, entityName);
        
        // 如果结果是List类型，则应用条件过滤
        if (result instanceof List) {
            List<Object> dataList = (List<Object>) result;
            
            // 应用条件过滤
            if (!paged.getConditions().isEmpty()) {
                // 这里可以调用服务层的过滤方法
                Map<String, Object> filteredResult = chromosomeDataService.queryChromosomeDataWithConditions(
                        sceneId, entityName, paged);
                dataList = (List<Object>) filteredResult.get("records");
            }
            
            return R.ok(dataList);
        } else {
            return R.ok(Collections.singletonList(result));
        }
    }
    
    /**
     * 通用接口，根据实体名称和ID查询Chromosome中的单个数据
     * 示例: /queryChromosome/order/get?sceneId=xxx&id=123
     *
     * @param sceneId 场景ID
     * @param entityName 实体名称 (如: order, entry, machine等)
     * @param id 数据ID
     * @return 单个数据对象
     */
    @GetMapping("/{entityName}/get")
    public R<Object> queryChromosomeDataById(
            @RequestParam String sceneId,
            @PathVariable String entityName,
            @RequestParam String id) {
        
        Object result = chromosomeDataService.queryChromosomeData(sceneId, entityName);
        
        // 如果结果是List类型，则根据ID查找特定项
        if (result instanceof List) {
            List<Object> dataList = (List<Object>) result;
            Object item = dataList.stream()
                    .filter(obj -> {
                        try {
                            // 尝试获取对象的id字段
                            java.lang.reflect.Field idField = obj.getClass().getDeclaredField("id");
                            idField.setAccessible(true);
                            Object itemId = idField.get(obj);
                            return itemId != null && itemId.toString().equals(id);
                        } catch (Exception e) {
                            // 如果没有id字段或访问失败，尝试使用Id字段
                            try {
                                java.lang.reflect.Field idField = obj.getClass().getDeclaredField("Id");
                                idField.setAccessible(true);
                                Object itemId = idField.get(obj);
                                return itemId != null && itemId.toString().equals(id);
                            } catch (Exception ex) {
                                return false;
                            }
                        }
                    })
                    .findFirst()
                    .orElse(null);
                    
            if (item != null) {
                return R.ok(item);
            } else {
                return R.failed("未找到ID为 " + id + " 的数据");
            }
        } else {
            // 如果不是列表，直接返回结果
            return R.ok(result);
        }
    }
    
//    /**
//     * 通用接口，根据实体名称更新Chromosome中的数据
//     * 示例: POST /queryChromosome/order/update
//     * Body: { "sceneId": "SCENE001", "data": {...} }
//     *
//     * @param entityName 实体名称 (如: order, entry, machine等)
//     * @param requestBody 包含sceneId和更新数据的请求体
//     * @return 更新结果
//     */
//    @PostMapping("/{entityName}/update")
//    public R<String> updateChromosomeData(
//            @PathVariable String entityName,
//            @RequestBody Map<String, Object> requestBody) {
//
//        String sceneId = (String) requestBody.get("sceneId");
//        @SuppressWarnings("unchecked")
//        Map<String, Object> data = (Map<String, Object>) requestBody.get("data");
//
//        if (sceneId == null || sceneId.isEmpty()) {
//            return R.failed("sceneId不能为空");
//        }
//
//        if (data == null || data.isEmpty()) {
//            return R.failed("更新数据不能为空");
//        }
//
//        boolean success = chromosomeDataService.updateChromosomeData(sceneId, entityName, data);
//        if (success) {
//            return R.ok("数据更新成功");
//        } else {
//            return R.failed("数据更新失败");
//        }
//    }
//
//    /**
//     * 通用接口，根据实体名称批量操作Chromosome中的数据
//     * 示例: POST /queryChromosome/order/batch
//     * Body: { "sceneId": "SCENE001", "data": [...] }
//     *
//     * @param entityName 实体名称 (如: order, entry, machine等)
//     * @param requestBody 包含sceneId和批量数据的请求体
//     * @return 批量操作结果
//     */
//    @PostMapping("/{entityName}/batch")
//    public R<String> batchChromosomeData(
//            @PathVariable String entityName,
//            @RequestBody Map<String, Object> requestBody) {
//
//        String sceneId = (String) requestBody.get("sceneId");
//        @SuppressWarnings("unchecked")
//        List<Map<String, Object>> data = (List<Map<String, Object>>) requestBody.get("data");
//
//        if (sceneId == null || sceneId.isEmpty()) {
//            return R.failed("sceneId不能为空");
//        }
//
//        if (data == null || data.isEmpty()) {
//            return R.failed("批量数据不能为空");
//        }
//
//        int successCount = chromosomeDataService.batchChromosomeData(sceneId, entityName, data);
//        return R.ok("批量操作成功，共处理 " + data.size() + " 条数据，成功更新 " + successCount + " 条");
//    }
//
//    /**
//     * 通用接口，根据实体名称删除Chromosome中的数据
//     * 示例: POST /queryChromosome/order/delete
//     * Body: { "sceneId": "SCENE001", "ids": [...] }
//     *
//     * @param entityName 实体名称 (如: order, entry, machine等)
//     * @param requestBody 包含sceneId和要删除的数据ID列表的请求体
//     * @return 删除操作结果
//     */
//    @PostMapping("/{entityName}/delete")
//    public R<String> deleteChromosomeData(
//            @PathVariable String entityName,
//            @RequestBody Map<String, Object> requestBody) {
//
//        String sceneId = (String) requestBody.get("sceneId");
//        @SuppressWarnings("unchecked")
//        List<Object> ids = (List<Object>) requestBody.get("ids");
//
//        if (sceneId == null || sceneId.isEmpty()) {
//            return R.failed("sceneId不能为空");
//        }
//
//        if (ids == null || ids.isEmpty()) {
//            return R.failed("删除ID列表不能为空");
//        }
//
//        int deleteCount = chromosomeDataService.deleteChromosomeData(sceneId, entityName, ids);
//        return R.ok("删除成功，共删除 " + ids.size() + " 条数据，实际删除 " + deleteCount + " 条");
//    }
}