package com.aps.service.Algorithm;

import com.aps.common.util.ProductionDeepCopyUtil;
import com.aps.entity.Algorithm.GAScheduleResult;
import com.aps.entity.Algorithm.ScheduleResultDetail;
import com.aps.entity.basic.*;
import com.aps.service.plan.AlgorithmScheduler8;
import com.aps.service.plan.MachineSchedulerService;
import com.baomidou.mybatisplus.core.toolkit.StringUtils;

import java.math.BigDecimal;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.time.temporal.ChronoUnit;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 作者：佟礼
 * 时间：2025-11-24
 */
public class MachineCalculator {
    private LocalDateTime baseTime;
    private final List<Machine> machines;
    private final MachineSchedulerService machineScheduler;

    public MachineCalculator(LocalDateTime baseTime, List<Machine> machines, MachineSchedulerService machineScheduler) {
        this.baseTime = baseTime;
        this.machines = machines;

        this.machineScheduler = machineScheduler;

    }



    /**
     * 获取机器下一个可用时间窗口（考虑班次约束）
     */
    public List<ScheduleResultDetail> getNextAvailableTime(Machine machine, int proposedStartTime,
                                                            int prevtime, int processingTime,
                                                            List<GAScheduleResult> existingTasks,
                                                            boolean isInterrupt, boolean istask,
                                                            boolean islockMachineTime) {
        LocalDateTime startTime = baseTime.plus(proposedStartTime, ChronoUnit.SECONDS);

        String prevtimestr = "";
        if (prevtime > -1) {
            prevtimestr = baseTime.plus(prevtime, ChronoUnit.SECONDS)
                    .format(DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss"));
        }

        // 查找合适的班次窗口
        return findEarliestStart(machine, processingTime, startTime, prevtimestr,
                existingTasks, istask, islockMachineTime);
    }


    // 查找最早可用开始时间

    private List<ScheduleResultDetail> findEarliestStart(
            Machine machine, int processingTime, LocalDateTime currentTime,
            String prevtime, List<GAScheduleResult> existingTasks, boolean checkprevtime, boolean islockMachineTime
    ) {
        List<GAScheduleResult> machineTasks = existingTasks.stream()
                .filter(t -> t.getMachineId() == machine.getId())
                .sorted(Comparator.comparingInt(GAScheduleResult::getStartTime))
                .collect(Collectors.toList());

        List<ScheduleResultDetail> times = new ArrayList<>();

        TimeSegment slot = GetCurrentOrNextShift(machine, currentTime, prevtime, checkprevtime);
        if (slot == null) return times;

        LocalDateTime prevTimeDateTime = StringUtils.isEmpty(prevtime) ? null : LocalDateTime.parse(prevtime);
        LocalDateTime startCandidate = slot.getStart().isAfter(prevTimeDateTime != null ? prevTimeDateTime : currentTime)
                ? slot.getStart()
                : (prevTimeDateTime != null ? prevTimeDateTime : currentTime);
        LocalDateTime endCandidate = startCandidate.plusSeconds(processingTime);

        if (endCandidate.isAfter(slot.getEnd())) {
            return CaldEarliestStart(machine, processingTime, currentTime, prevtime, machineTasks, checkprevtime,islockMachineTime);
        } else {
            ScheduleResultDetail time = new ScheduleResultDetail();
            time.setKey(slot.getKey());
            time.setStartTime((int) ChronoUnit.SECONDS.between(baseTime, startCandidate));
            time.setEndTime((int) ChronoUnit.SECONDS.between(baseTime, endCandidate));
            times.add(time);
            if(islockMachineTime) {
                RemoveMachineAvailable(machine, time);
            }
            return times;
        }
    }


    private List<ScheduleResultDetail> CaldEarliestStart(
            Machine machine, int processingTime, LocalDateTime currentTime,
            String prevtime, List<GAScheduleResult> machineTasks, boolean checkprevtime, boolean islockMachineTime
    ) {
        int remainingTime = processingTime;
        LocalDateTime st = StringUtils.isEmpty(prevtime) ? currentTime : LocalDateTime.parse(prevtime);
        LocalDateTime prevEnd = LocalDateTime.of(2000, 1, 1, 0, 0, 0);
        List<ScheduleResultDetail> times = new ArrayList<>();
        List<ScheduleResultDetail> oldTimes = new ArrayList<>();

        while (remainingTime > 0) {
            TimeSegment shift = GetCurrentOrNextShift(machine, currentTime, prevtime, checkprevtime);
            if (shift == null) break;

            LocalDateTime shiftStart = shift.getStart();
            LocalDateTime shiftEnd = shift.getEnd();

            // 检查班次间冲突
            if (!prevEnd.isEqual(LocalDateTime.of(2000, 1, 1, 0, 0, 0))) {
                if (prevEnd.isBefore(currentTime)) {
                    // 检查班次间任务
                    LocalDateTime finalPrevEnd = prevEnd;
                    boolean hasTask = machineTasks.stream()
                            .anyMatch(t -> {
                                LocalDateTime taskStart = baseTime.plusSeconds(t.getStartTime());
                                return taskStart.isAfter(finalPrevEnd) && taskStart.isBefore(shiftStart);
                            });

                    if (hasTask) {
                        // 重置状态
                        currentTime = shiftStart;
                        st = shiftStart;
                        remainingTime = processingTime;
                        prevEnd = LocalDateTime.of(2000, 1, 1, 0, 0, 0);
                        oldTimes.addAll(ProductionDeepCopyUtil.deepCopyList(times));
                        times.clear();
                        continue;
                    }

                    // 检查班次间维修窗口
                    if (machine.getMaintenanceWindows() != null) {
                        LocalDateTime finalPrevEnd1 = prevEnd;
                        boolean hasMaintenance = machine.getMaintenanceWindows().stream()
                                .anyMatch(w -> w.getStartTime().isAfter(finalPrevEnd1) && w.getStartTime().isBefore(shiftStart));

                        if (hasMaintenance) {
                            currentTime = shiftStart;
                            st = shiftStart;
                            remainingTime = processingTime;
                            prevEnd = LocalDateTime.of(2000, 1, 1, 0, 0, 0);
                            times.clear();
                            continue;
                        }
                    }
                }
            }

            prevEnd = shiftEnd;
            // 计算有效时间
            LocalDateTime effectiveStart = st.isAfter(shiftStart) ? st : shiftStart;
            long availableSeconds = ChronoUnit.SECONDS.between(effectiveStart, shiftEnd);

            // 处理当前班次
            int processable = Math.min(remainingTime, (int) availableSeconds);
            remainingTime -= processable;
            currentTime = effectiveStart.plusSeconds(processable);

            // 添加时间详情
            ScheduleResultDetail time = new ScheduleResultDetail();
            time.setKey(shift.getKey());
            time.setStartTime((int) ChronoUnit.SECONDS.between(baseTime, effectiveStart));
            time.setEndTime((int) ChronoUnit.SECONDS.between(baseTime, currentTime));
            times.add(time);
            if (islockMachineTime) {
                // 还原未使用的时间段
                RemoveMachineAvailable(machine, time);
            }

        }

        // 还原未使用的时间段
        if (islockMachineTime) {
            // 还原未使用的时间段
            AddMachineAvailable(machine, oldTimes);
        }
        return times;
    }

    /**
     * 获取设备当前或下一个有效班次
     */
    private TimeSegment GetCurrentOrNextShift(Machine machine, LocalDateTime time, String prevtime, boolean checkprevtime) {
        TimeSegment start = null;

        // 查找有效班次
        start = machine.getAvailability().stream()
                .filter(slot -> !slot.isUsed() && slot.getType() != SegmentType.MAINTENANCE)
                .filter(slot -> slot.getStart().isAfter(time) || slot.getEnd().isAfter(time))
                .findFirst()
                .orElse(null);

        if (start == null) {
            // 生成新时间段
            List<TimeSegment> timeSegments = machineScheduler.generateTimeSegment(machine, time.plusDays(1));
            machine.getAvailability().addAll(timeSegments);

            // 更新设备时间线
            Machine originalMachine = machines.stream()
                    .filter(t -> t.getId() == machine.getId())
                    .findFirst()
                    .orElse(null);
            if (originalMachine != null) {
                MachineTimeline timeline = machineScheduler.getOrCreateTimeline(originalMachine);
            }

            // 递归查找
            return GetCurrentOrNextShift(machine, time, prevtime, checkprevtime);
        }

        return start;
    }



    private void RemoveMachineAvailable(Machine machine, ScheduleResultDetail geneDetails) {
        List<TimeSegment> timeSegments = new ArrayList<>();

        int index = machine.getAvailability().stream()
                .filter(t -> t.getKey().equals(geneDetails.getKey()))
                .findFirst()
                .map(machine.getAvailability()::indexOf)
                .orElse(-1);

        if (index > -1) {
            TimeSegment targetSegment = machine.getAvailability().get(index);
            LocalDateTime geneEndTime = baseTime.plusSeconds(geneDetails.getEndTime());

            if (targetSegment.getEnd().isAfter(geneEndTime)) {
                TimeSegment usedSegment = new TimeSegment();
                usedSegment.setStart(baseTime.plusSeconds(geneDetails.getStartTime()));
                usedSegment.setEnd(geneEndTime);
                usedSegment.setHoliday(false);
                usedSegment.setKey(UUID.randomUUID().toString());
                usedSegment.setType(SegmentType.REGULAR);
                usedSegment.setUsed(true);
                timeSegments.add(usedSegment);

                geneDetails.setKey(usedSegment.getKey());
                targetSegment.setStart(geneEndTime);
            } else {
                targetSegment.setUsed(true);
            }
        }

        if (!timeSegments.isEmpty()) {
            machine.getAvailability().addAll(timeSegments);
        }
        machine.getAvailability().sort(Comparator.comparing(TimeSegment::getStart));
    }

    public void AddMachineAvailable(Machine machine, List<ScheduleResultDetail> geneDetails) {
        if (geneDetails == null || geneDetails.isEmpty()) return;

        for (ScheduleResultDetail detail : geneDetails) {
            machine.getAvailability().stream()
                    .filter(t -> t.getKey().equals(detail.getKey()))
                    .findFirst()
                    .ifPresent(t -> t.setUsed(false));
        }

        machine.setAvailability(MergeSegments(machine.getAvailability()));
    }

    private List<TimeSegment> MergeSegments(List<TimeSegment> segments) {
        List<TimeSegment> maintenanceSegments = segments.stream()
                .filter(t -> t.getType() == SegmentType.MAINTENANCE)
                .collect(Collectors.toList());

        List<TimeSegment> unusedRegularSegments = segments.stream()
                .filter(t -> t.getType() != SegmentType.MAINTENANCE && !t.isUsed())
                .collect(Collectors.toList());

        List<TimeSegment> usedRegularSegments = segments.stream()
                .filter(t -> t.getType() != SegmentType.MAINTENANCE && t.isUsed())
                .collect(Collectors.toList());

        unusedRegularSegments.sort(Comparator.comparing(TimeSegment::getStart));

        Deque<TimeSegment> mergedUnused = new ArrayDeque<>();
        for (TimeSegment seg : unusedRegularSegments) {
            if (!mergedUnused.isEmpty() && mergedUnused.peekLast().getEnd().isAfter(seg.getStart())) {
                TimeSegment last = mergedUnused.pollLast();
                last.setEnd(last.getEnd().isAfter(seg.getEnd()) ? last.getEnd() : seg.getEnd());
                mergedUnused.offerLast(last);
            } else {
                mergedUnused.offerLast(seg);
            }
        }

        List<TimeSegment> result = new ArrayList<>(mergedUnused);
        result.addAll(usedRegularSegments);
        result.addAll(maintenanceSegments);
        result.sort(Comparator.comparing(TimeSegment::getStart));

        return result;
    }


}
