package com.example.controller.codeGeneratorController;


import com.example.entity.codeGeneratorEntity.DatabaseConfig;
import com.example.entity.codeGeneratorEntity.GenerateRequest;
import com.example.service.codeGeneratorService.CodeGeneratorService;
import com.example.service.codeGeneratorService.DatabaseConfigService;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import java.util.List;
import java.util.Map;

@RestController
@RequestMapping("/api/code-generator")
@Api(tags = "代码生成器")
public class CodeGeneratorController {

    @Autowired
    private CodeGeneratorService codeGeneratorService;

    @Autowired
    private DatabaseConfigService databaseConfigService;

    @GetMapping("/config")
    @ApiOperation("获取当前主数据源配置")
    public DatabaseConfig getDatabaseConfig() {
        DatabaseConfig config = databaseConfigService.getPrimaryDatabaseConfig();
        if (config == null) {
            throw new RuntimeException("无法获取数据库配置，请检查application.yml配置");
        }
        return config;
    }

    @GetMapping("/config/{dataSourceName}")
    @ApiOperation("获取指定数据源配置")
    public DatabaseConfig getDatabaseConfig(@PathVariable String dataSourceName) {
        DatabaseConfig config = databaseConfigService.getDatabaseConfig(dataSourceName);
        if (config == null) {
            throw new RuntimeException("数据源 " + dataSourceName + " 配置不存在或配置错误");
        }
        return config;
    }

    @GetMapping("/tables/{dataSourceName}")
    @ApiOperation("获取指定数据源的表列表")
    public List<String> getTables(@PathVariable String dataSourceName) {
        DatabaseConfig config = databaseConfigService.getDatabaseConfig(dataSourceName);
        if (config == null || config.getUrl() == null) {
            throw new RuntimeException("数据源 " + dataSourceName + " 配置错误，无法获取表列表");
        }
        return codeGeneratorService.getTableList(config);
    }

    @GetMapping("/test-connection/{dataSourceName}")
    @ApiOperation("测试指定数据源连接")
    public String testConnection(@PathVariable String dataSourceName) {
        DatabaseConfig config = databaseConfigService.getDatabaseConfig(dataSourceName);
        if (config == null) {
            return dataSourceName + " 数据源配置不存在";
        }
        
        boolean isConnected = databaseConfigService.testConnection(config);
        return isConnected ? dataSourceName + " 数据库连接成功！" : dataSourceName + " 数据库连接失败！";
    }

    @PostMapping("/generate/{dataSourceName}")
    @ApiOperation("使用指定数据源生成代码")
    public String generateCode(@PathVariable String dataSourceName, 
                               @RequestBody GenerateRequest request) {
        try {
            DatabaseConfig dbConfig = databaseConfigService.getDatabaseConfig(dataSourceName);
            if (dbConfig == null || dbConfig.getUrl() == null) {
                return "数据源 " + dataSourceName + " 配置错误，请检查application.yml";
            }
            
            codeGeneratorService.generateCode(dbConfig, request.getTables(), 
                request.getPackageName(), request.getAuthor());
            return "代码生成成功！(数据源: " + dataSourceName + ")";
        } catch (Exception e) {
            return "代码生成失败: " + e.getMessage();
        }
    }
}